# C2PA Content Credentials Viewer Plugin
## Technische Spezifikation & Dokumentation

**Version:** 4.9.0  
**Autor:** Thomas Brandt  
**Website:** https://www.thomasbrandt.eu  
**Based on:** [Content Credentials](https://contentcredentials.org/)

---

## 📋 Inhaltsverzeichnis

1. [Übersicht](#übersicht)
2. [Technische Architektur](#technische-architektur)
3. [Plugin-Struktur](#plugin-struktur)
4. [Funktionsweise](#funktionsweise)
5. [C2PA-Integration](#c2pa-integration)
6. [Frontend-Implementierung](#frontend-implementierung)
7. [Admin-Interface](#admin-interface)
8. [Datenverarbeitung](#datenverarbeitung)
9. [Sicherheit](#sicherheit)
10. [Performance](#performance)
11. [Kompatibilität](#kompatibilität)
12. [API & Hooks](#api--hooks)
13. [Konfiguration](#konfiguration)
14. [Troubleshooting](#troubleshooting)

---

## 🎯 Übersicht

Das **C2PA Content Credentials Viewer Plugin** ist eine WordPress-Erweiterung, die Content Authenticity Initiative (CAI) C2PA-Manifeste von Bildern ausliest und analysiert. Es bietet eine vollständige Transparenz über die Authentizität und Bearbeitungshistorie von Bildern.

### Hauptfunktionen
- **Automatische C2PA-Erkennung** auf allen Seiten
- **Detaillierte Metadaten-Analyse** (EXIF, IPTC, XMP)
- **Bearbeitungshistorie** mit Photoshop-Aktionen
- **KI-Herkunftserkennung** (Midjourney, DALL-E, etc.)
- **Zweisprachige Benutzeroberfläche** (Deutsch/Englisch)
- **Responsive Design** mit Glassmorphism-Stil
- **Benutzerfreundliche Admin-Konfiguration**

---

## 🏗️ Technische Architektur

### Systemanforderungen
- **WordPress:** 5.0+
- **PHP:** 7.4+
- **Browser:** Moderne Browser mit ES6+ Support
- **C2PA-Bibliothek:** 0.30.11 (CDN-geladen)

### Technologie-Stack
```
Frontend:
├── Vanilla JavaScript (ES6+)
├── CSS3 (Glassmorphism, Flexbox, Grid)
├── HTML5 (Semantic Markup)
└── C2PA Web SDK (CDN)

Backend:
├── PHP 7.4+
├── WordPress Hooks & Filters
├── WordPress Settings API
└── WordPress Admin API

Externe Dienste:
├── C2PA CDN (jsdelivr.net)
├── EXIFR CDN (exifr.js)
└── Content Credentials API
```

---

## 📁 Plugin-Struktur

```
wp-c2pa-viewer-combined/
├── wp-c2pa-viewer-combined.php          # Hauptdatei
├── assets/
│   ├── css/
│   │   ├── c2pa-viewer-combined.css     # Frontend-Styles
│   │   ├── admin.css                    # Admin-Styles
│   │   └── notifications.css            # Notification-Styles
│   ├── js/
│   │   ├── c2pa-viewer-combined.js      # Frontend-Logic
│   │   └── admin.js                     # Admin-Logic
│   └── images/
│       └── README.md                    # Bild-Assets-Dokumentation
└── TECHNICAL_SPECS.md                   # Diese Dokumentation
```

### Datei-Funktionen

#### Hauptdatei (`wp-c2pa-viewer-combined.php`)
- Plugin-Initialisierung
- WordPress-Hooks-Registrierung
- Admin-Interface-Setup
- Settings-Management
- Asset-Loading-Logic

#### Frontend-JavaScript (`c2pa-viewer-combined.js`)
- C2PA-Manifest-Parsing
- Bild-Scanning-Logic
- Modal-Popup-Management
- Übersetzungs-System
- Notification-System

#### Frontend-CSS (`c2pa-viewer-combined.css`)
- Glassmorphism-Design
- Responsive Layout
- Modal-Styling
- Badge-Design
- Animationen

---

## ⚙️ Funktionsweise

### 1. Plugin-Initialisierung
```php
class WP_C2PA_Viewer_Combined {
    public function __construct() {
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        // ... weitere Hooks
    }
}
```

### 2. Asset-Loading
```php
public function enqueue_scripts() {
    if (!$this->should_load_assets()) {
        return; // Nur laden wenn nötig
    }
    
    wp_enqueue_style('wp-c2pa-combined', ...);
    wp_enqueue_script('wp-c2pa-combined', ...);
    wp_localize_script('wp-c2pa-combined', 'WPC2PACombinedConfig', ...);
}
```

### 3. Frontend-Scanning
```javascript
function scan() {
    const nodes = Array.from(document.querySelectorAll('img,video'))
        .filter(n => !n.dataset.wpC2paCombinedProcessed);
    nodes.forEach(n => processEl(n));
}
```

### 4. C2PA-Verarbeitung
```javascript
async function processEl(el) {
    const [man, meta] = await Promise.all([
        readManifest(el), 
        readMetadata(el)
    ]);
    attachBadge(el, man, meta);
}
```

---

## 🔍 C2PA-Integration

### C2PA-Bibliothek
- **Version:** 0.30.11
- **Quelle:** CDN (jsdelivr.net)
- **Lazy Loading:** Nur bei Bedarf geladen

### Manifest-Parsing
```javascript
async function readManifest(el) {
    if (!cfg.ready) await loadC2PA();
    
    const src = chooseBestUrl(el);
    if (!src) return null;
    
    try {
        const manifest = await cfg.c2pa.read(src);
        return manifest;
    } catch (e) {
        return null;
    }
}
```

### Metadaten-Extraktion
```javascript
async function readMetadata(el) {
    const exifr = await loadExifr();
    const src = chooseBestUrl(el);
    
    try {
        const metadata = await exifr.parse(src);
        return processMetadata(metadata);
    } catch (e) {
        return null;
    }
}
```

### Unterstützte Formate
- **Bilder:** JPEG, PNG, WebP, TIFF, HEIC
- **Videos:** MP4, MOV, AVI (begrenzt)
- **Metadaten:** EXIF, IPTC, XMP, C2PA

---

## 🎨 Frontend-Implementierung

### Glassmorphism-Design
```css
.wp-c2pa-combined-modal {
    background: rgba(31, 41, 55, 0.8);
    backdrop-filter: blur(30px) saturate(180%);
    border: 1px solid rgba(255, 255, 255, 0.1);
    border-radius: 24px;
    box-shadow: 0 25px 50px rgba(0, 0, 0, 0.25);
}
```

### Responsive Layout
```css
@media (max-width: 768px) {
    .wp-c2pa-combined-modal {
        width: 95vw;
        margin: 2vh auto 0;
    }
}
```

### Badge-System
```javascript
function attachBadge(el, man, meta) {
    const badge = createBadge(man, meta);
    const wrapper = createWrapper(el);
    wrapper.appendChild(badge);
}
```

### Modal-Popup
```javascript
function ensureModal() {
    if (document.getElementById('wp-c2pa-combined-modal')) return;
    
    const modal = document.createElement('div');
    modal.id = 'wp-c2pa-combined-modal';
    modal.innerHTML = createModalHTML();
    document.body.appendChild(modal);
}
```

---

## ⚙️ Admin-Interface

### Settings-Struktur
```php
$settings_sections = [
    'wp_c2pa_combined_language' => 'Language Settings',
    'wp_c2pa_combined_main' => 'Main Settings',
    'wp_c2pa_combined_badge' => 'Badge Settings',
    'wp_c2pa_combined_pages' => 'Page Settings'
];
```

### Sprachsystem
```php
function get_section_title($section) {
    $current_language = $this->get_option('language', 'de');
    $translations = [
        'de' => ['main' => 'Haupteinstellungen'],
        'en' => ['main' => 'Main Settings']
    ];
    return $translations[$current_language][$section] ?? $section;
}
```

### Einstellungsfelder
- **Sprachauswahl:** Deutsch/Englisch
- **Badge-Position:** 4 Positionen verfügbar
- **Badge-Stil:** Icon, Text, Icon+Text
- **Seiten-Ausschluss:** URLs und Post-Types
- **Benachrichtigungen:** Ein/Aus
- **Titelbilder:** Ausschluss-Option

---

## 📊 Datenverarbeitung

### C2PA-Daten-Struktur
```javascript
const manifest = {
    title: "Bildtitel",
    creator: { name: "Ersteller" },
    signatureInfo: { time: "2024-01-01T12:00:00Z" },
    claimGenerator: "Adobe Photoshop 26.8.1",
    actions: [
        {
            action: "c2pa.adjustments",
            parameters: { description: "Helligkeit erhöht" }
        }
    ],
    ingredients: [
        {
            title: "Ebene 1",
            format: "image/jpeg",
            hash: "sha256:..."
        }
    ]
};
```

### Metadaten-Verarbeitung
```javascript
function processMetadata(raw) {
    return {
        summary: {
            fileName: extractFileName(raw),
            dimensions: extractDimensions(raw),
            camera: extractCamera(raw),
            software: extractSoftware(raw),
            createdAt: extractCreationTime(raw)
        },
        raw: raw
    };
}
```

### KI-Erkennung
```javascript
function detectAI(rawDataString) {
    const patterns = {
        midjourney: /midjourney/i,
        openai: /openai|dall-e/i,
        adobeFirefly: /adobe_firefly/i,
        canva: /canva/i
    };
    
    return Object.entries(patterns)
        .filter(([key, pattern]) => pattern.test(rawDataString))
        .map(([key]) => key);
}
```

---

## 🔒 Sicherheit

### XSS-Schutz
```php
echo esc_html($value);
echo esc_attr($attribute);
echo wp_kses_post($html_content);
```

### CSRF-Schutz
```php
wp_nonce_field('wp_c2pa_combined_options');
wp_verify_nonce($_POST['_wpnonce'], 'wp_c2pa_combined_options');
```

### Datei-Validierung
```javascript
function isValidImageUrl(url) {
    const validExtensions = /\.(jpe?g|png|webp|gif|tiff?|bmp|heic|heif|svg)$/i;
    return validExtensions.test(url);
}
```

### Berechtigungsprüfung
```php
if (!current_user_can('manage_options')) {
    return;
}
```

---

## ⚡ Performance

### Lazy Loading
- C2PA-Bibliothek nur bei Bedarf geladen
- Bilder werden einzeln verarbeitet
- Modal wird erst bei Klick erstellt

### Caching
```javascript
const cache = new Map();
function getCachedResult(key) {
    if (cache.has(key)) {
        return cache.get(key);
    }
    // Verarbeitung und Caching
}
```

### Optimierungen
- **MutationObserver:** Dynamische Inhalte
- **RequestAnimationFrame:** Smooth Animations
- **Debouncing:** Häufige Events
- **Asset-Minifizierung:** Komprimierte Dateien

### Performance-Metriken
- **Ladezeit:** < 100ms (ohne C2PA)
- **C2PA-Parsing:** < 500ms pro Bild
- **Memory-Usage:** < 5MB zusätzlich
- **CPU-Impact:** < 2% bei normaler Nutzung

---

## 🔧 Kompatibilität

### Browser-Support
- **Chrome:** 88+
- **Firefox:** 85+
- **Safari:** 14+
- **Edge:** 88+

### WordPress-Versionen
- **Minimum:** 5.0
- **Getestet:** 6.4
- **Empfohlen:** 6.0+

### PHP-Versionen
- **Minimum:** 7.4
- **Empfohlen:** 8.0+
- **Getestet:** 8.2

### Theme-Kompatibilität
- **Standard-Themes:** Vollständig kompatibel
- **Custom Themes:** Abhängig von CSS-Konflikten
- **Page Builder:** Kompatibel mit Elementor, Divi, etc.

---

## 🔌 API & Hooks

### WordPress-Hooks
```php
// Actions
add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
add_action('admin_menu', [$this, 'add_admin_menu']);
add_action('admin_init', [$this, 'register_settings']);

// Filters
add_filter('big_image_size_threshold', [$this, 'disable_image_scaling']);
```

### Custom Hooks
```php
// Vor C2PA-Verarbeitung
do_action('wp_c2pa_before_process', $image_url);

// Nach C2PA-Verarbeitung
do_action('wp_c2pa_after_process', $manifest, $metadata);

// Badge-Erstellung
apply_filters('wp_c2pa_badge_content', $badge_html, $manifest);
```

### JavaScript-API
```javascript
// Globale Funktionen
window.WPC2PACombinedViewer = {
    scan: function() { /* ... */ },
    processEl: function(el) { /* ... */ }
};

// Event-Listener
document.addEventListener('wp-c2pa-credentials-found', function(e) {
    console.log('C2PA gefunden:', e.detail);
});
```

---

## ⚙️ Konfiguration

### Standard-Einstellungen
```php
$default_settings = [
    'language' => 'de',
    'auto_scan' => true,
    'badge_display' => 'icon_text',
    'badge_position' => 'bottom-left',
    'badge_size' => 'medium',
    'show_notifications' => true,
    'exclude_featured_images' => false,
    'disable_notifications' => false
];
```

### Konfigurationsdatei
```javascript
const WPC2PACombinedConfig = {
    version: '4.9.0',
    language: 'de',
    settings: {
        autoScan: true,
        badgeDisplay: 'icon_text',
        badgePosition: 'bottom-left',
        // ... weitere Einstellungen
    }
};
```

### Umgebungsvariablen
```php
// Optional: C2PA-Version überschreiben
define('WP_C2PA_VERSION', '0.30.11');

// Optional: Debug-Modus
define('WP_C2PA_DEBUG', false);
```

---

## 🛠️ Troubleshooting

### Häufige Probleme

#### 1. C2PA-Bibliothek lädt nicht
```javascript
// Lösung: Manueller Fallback
if (!window.c2pa) {
    console.warn('C2PA-Bibliothek nicht verfügbar');
    // Fallback-Logic implementieren
}
```

#### 2. Bilder werden nicht erkannt
```php
// Prüfung: Debug-Logging aktivieren
if (defined('WP_C2PA_DEBUG') && WP_C2PA_DEBUG) {
    error_log('C2PA Debug: Bild verarbeitet - ' . $image_url);
}
```

#### 3. Admin-Einstellungen speichern nicht
```php
// Lösung: Nonce-Validierung prüfen
if (!wp_verify_nonce($_POST['_wpnonce'], 'wp_c2pa_combined_options')) {
    wp_die('Sicherheitscheck fehlgeschlagen');
}
```

#### 4. CSS-Konflikte
```css
/* Lösung: Spezifische Selektoren verwenden */
.wp-c2pa-combined-modal {
    z-index: 9999 !important;
}
```

### Debug-Modus
```php
// Aktivierung
define('WP_C2PA_DEBUG', true);

// Debug-Logging
if (defined('WP_C2PA_DEBUG') && WP_C2PA_DEBUG) {
    error_log('C2PA Debug: ' . $message);
}
```

### Performance-Monitoring
```javascript
// Performance-Messung
const startTime = performance.now();
await processImage(image);
const endTime = performance.now();
console.log(`Verarbeitungszeit: ${endTime - startTime}ms`);
```

---

## 📈 Zukünftige Entwicklungen

### Geplante Features
- **Video-C2PA-Support:** Erweiterte Video-Analyse
- **Batch-Processing:** Massenverarbeitung von Bildern
- **API-Integration:** REST-API für externe Anwendungen
- **Erweiterte KI-Erkennung:** Mehr KI-Tools unterstützen
- **Export-Funktionen:** PDF-Reports generieren

### Technische Verbesserungen
- **WebAssembly:** C2PA-Parsing optimieren
- **Service Worker:** Offline-Funktionalität
- **Progressive Web App:** PWA-Features
- **Web Components:** Modulare Komponenten

### Performance-Optimierungen
- **Web Workers:** Hintergrund-Verarbeitung
- **IndexedDB:** Lokale Datenspeicherung
- **HTTP/2:** Optimierte Asset-Delivery
- **CDN-Integration:** Globale Verteilung

---

## 📞 Support & Kontakt

### Technischer Support
- **Entwickler:** Thomas Brandt
- **Website:** https://www.thomasbrandt.eu
- **GitHub:** [Repository-Link]
- **Dokumentation:** Diese Spezifikation

### Community
- **WordPress.org:** Plugin-Forum
- **Content Credentials:** https://contentcredentials.org/
- **C2PA-Spezifikation:** https://c2pa.org/

### Lizenz
- **Lizenz:** GPLv2 oder später
- **Kommerzielle Nutzung:** Erlaubt
- **Modifikationen:** Erlaubt mit Attribution

---

## 📋 Changelog

### Version 4.9.0
- ✅ Vollständige Zweisprachigkeit (Deutsch/Englisch)
- ✅ Glassmorphism-Design implementiert
- ✅ Erweiterte KI-Erkennung
- ✅ Verbesserte Performance
- ✅ Admin-Interface modernisiert
- ✅ Notification-System überarbeitet
- ✅ Responsive Design optimiert

### Version 4.8.0
- ✅ C2PA-Integration implementiert
- ✅ Basis-Frontend-Funktionalität
- ✅ Admin-Settings-System
- ✅ Badge-System
- ✅ Modal-Popup

---

**Dokumentation erstellt:** 27. August 2025  
**Letzte Aktualisierung:** 27. August 2025  
**Version:** 4.9.0
