/* WP C2PA Admin Panel JavaScript */
jQuery(document).ready(function($) {
    'use strict';

    // Initialize color pickers
    $('.color-picker').wpColorPicker({
        change: function(event, ui) {
            updateBadgePreview();
        },
        clear: function() {
            updateBadgePreview();
        }
    });

    // Icon mappings for preview
    const iconMap = {
        'shield': '🛡️',
        'check': '✅',
        'lock': '🔒',
        'star': '⭐',
        'verified': '✓',
        'certificate': '📜',
        'eye': '👁️',
        'info': 'ℹ️',
        'badge': '🏷️',
        'security': '🔐',
        'eye-shield': '<img src="' + pluginUrl + 'assets/images/eye_shield_badge_transparent.svg" style="width: 16px; height: 16px; vertical-align: middle;" alt="Eye Shield">'
    };

    // Position mappings
    const positionMap = {
        'bottom-left': { bottom: '10px', left: '10px', top: 'auto', right: 'auto' },
        'bottom-right': { bottom: '10px', right: '10px', top: 'auto', left: 'auto' },
        'top-left': { top: '10px', left: '10px', bottom: 'auto', right: 'auto' },
        'top-right': { top: '10px', right: '10px', bottom: 'auto', left: 'auto' }
    };

    // Update badge preview function
    function updateBadgePreview() {
        const $preview = $('#badge-preview');
        
        if ($preview.length === 0) return;

        // Get current values
        const text = $('input[data-setting="badge_text"]').val() || 'Content Credentials';
        const color = $('input[data-setting="badge_color"]').val() || '#0073aa';
        const icon = $('input[data-setting="badge_icon"]:checked').val() || 'shield';
        const size = $('select[data-setting="badge_size"]').val() || 'medium';
        const style = $('select[data-setting="badge_style"]').val() || 'solid';
        const position = $('select[data-setting="badge_position"]').val() || 'bottom-left';
        const display = $('select[data-setting="badge_display"]').val() || 'icon_text';
        const opacity = $('select[data-setting="badge_opacity"]').val() || '1.0';
        const shadow = $('select[data-setting="badge_shadow"]').val() || 'light';
        const borderRadius = $('select[data-setting="badge_border_radius"]').val() || '8';
        const padding = $('select[data-setting="badge_padding"]').val() || '8';
        const fontSize = $('select[data-setting="badge_font_size"]').val() || '14';
        const fontWeight = $('select[data-setting="badge_font_weight"]').val() || '600';

        // Update text content based on display setting
        const iconSymbol = iconMap[icon] || '🛡️';
        
        // Spezielle Behandlung für eye-shield SVG
        let displayContent;
        if (icon === 'eye-shield') {
            // Für eye-shield SVG verwenden wir innerHTML
            switch(display) {
                case 'icon_only':
                    displayContent = iconSymbol;
                    break;
                case 'text_only':
                    displayContent = text;
                    break;
                case 'icon_text':
                default:
                    displayContent = iconSymbol + ' ' + text;
                    break;
            }
            $preview.html(displayContent);
        } else {
            // Für normale Emoticons verwenden wir textContent
            switch(display) {
                case 'icon_only':
                    displayContent = iconSymbol;
                    break;
                case 'text_only':
                    displayContent = text;
                    break;
                case 'icon_text':
                default:
                    displayContent = iconSymbol + ' ' + text;
                    break;
            }
            $preview.text(displayContent);
        }
        
        // CSS-Styling basierend auf Display-Einstellung
        switch(display) {
            case 'icon_only':
                $preview.css({
                    'padding': '6px',
                    'min-width': 'auto',
                    'width': 'auto'
                });
                break;
            case 'text_only':
                $preview.css({
                    'padding': padding + 'px',
                    'min-width': 'auto',
                    'width': 'auto'
                });
                break;
            case 'icon_text':
            default:
                $preview.css({
                    'padding': padding + 'px',
                    'min-width': 'auto',
                    'width': 'auto'
                });
                break;
        }

        // Update position
        const pos = positionMap[position];
        $preview.css(pos);

        // Update color
        $preview[0].style.setProperty('--badge-color', color);

        // Update style classes
        $preview.removeClass('style-outline style-minimal size-small size-large');
        $preview.addClass('style-' + style);
        $preview.addClass('size-' + size);
        
        // Apply size-specific styling to preview (only font size, padding is handled by display setting)
        switch(size) {
            case 'small':
                $preview.css('font-size', '10px');
                break;
            case 'medium':
                $preview.css('font-size', '12px');
                break;
            case 'large':
                $preview.css('font-size', '14px');
                break;
        }

        // Apply specific styles
        switch (style) {
            case 'outline':
                $preview.css({
                    'background': 'transparent',
                    'border': '2px solid ' + color,
                    'color': color
                });
                break;
            case 'minimal':
                $preview.css({
                    'background': 'rgba(255, 255, 255, 0.9)',
                    'border': '1px solid rgba(0, 0, 0, 0.1)',
                    'color': color
                });
                break;
            default: // solid
                $preview.css({
                    'background': color,
                    'border': 'none',
                    'color': 'white'
                });
                break;
        }
        
        // Apply new styling options
        $preview.css({
            'opacity': opacity,
            'border-radius': borderRadius + 'px',
            'padding': padding + 'px',
            'font-size': fontSize + 'px',
            'font-weight': fontWeight
        });
        
        // Apply shadow effects
        switch (shadow) {
            case 'none':
                $preview.css('box-shadow', 'none');
                break;
            case 'light':
                $preview.css('box-shadow', '0 2px 4px rgba(0,0,0,0.1)');
                break;
            case 'medium':
                $preview.css('box-shadow', '0 4px 8px rgba(0,0,0,0.15)');
                break;
            case 'strong':
                $preview.css('box-shadow', '0 8px 16px rgba(0,0,0,0.2)');
                break;
        }
        

    }

    // Listen for changes to badge settings
    $(document).on('change input', '.badge-setting', function() {
        updateBadgePreview();
    });

    // Real-time text input updates
    $(document).on('keyup', 'input[data-setting="badge_text"]', function() {
        updateBadgePreview();
    });

    // Initialize preview on page load
    updateBadgePreview();

    // Reset settings button
    $('#reset-settings').on('click', function(e) {
        e.preventDefault();
        
        if (confirm('Sind Sie sicher, dass Sie alle Einstellungen auf die Standardwerte zurücksetzen möchten?')) {
            // Reset to default values
            $('input[data-setting="badge_text"]').val('Content Credentials');
            $('input[data-setting="badge_color"]').val('#0073aa');
            $('input[data-setting="badge_icon"][value="shield"]').prop('checked', true);
            $('select[data-setting="badge_size"]').val('medium');
            $('select[data-setting="badge_style"]').val('solid');
            $('select[data-setting="badge_position"]').val('bottom-left');
            
            // Update color picker
            $('.color-picker').wpColorPicker('color', '#0073aa');
            
            // Update preview
            updateBadgePreview();
            
            // Show notification
            showNotification('Einstellungen wurden zurückgesetzt', 'info');
        }
    });

    // Form submission with validation
    $('#c2pa-settings-form').on('submit', function(e) {
        const text = $('input[data-setting="badge_text"]').val().trim();
        const display = $('select[data-setting="badge_display"]').val() || 'icon_text';
        
        // Only validate text if not "icon_only"
        if (display !== 'icon_only' && text === '') {
            e.preventDefault();
            alert('Badge-Text darf nicht leer sein.');
            $('input[data-setting="badge_text"]').focus();
            return false;
        }
        
        showNotification('Einstellungen werden gespeichert...', 'info');
    });

    // Status monitoring
    function updateStatus() {
        // This would normally check actual plugin status
        // For demo purposes, we'll simulate active status
        $('.wp-c2pa-status-item').each(function() {
            $(this).find('.status-indicator').addClass('active');
        });
    }

    // Enhanced settings section animations
    $('.wp-c2pa-admin-section h2').on('click', function() {
        const $section = $(this).parent();
        const $content = $section.find('table, .form-table');
        
        $content.slideToggle(300);
        $(this).toggleClass('collapsed');
    });

    // Notification system
    function showNotification(message, type = 'info') {
        // Remove existing notifications
        $('.settings-saved-notice').remove();
        
        const $notice = $('<div class="settings-saved-notice"></div>')
            .text(message)
            .addClass('notice-' + type);
        
        $('.wp-c2pa-admin-main').prepend($notice);
        $notice.slideDown(300);
        
        // Auto-hide after 3 seconds
        setTimeout(function() {
            $notice.slideUp(300, function() {
                $notice.remove();
            });
        }, 3000);
    }

    // Shortcode copy functionality
    $(document).on('click', 'code', function() {
        const text = $(this).text();
        
        // Create temporary textarea for copying
        const $temp = $('<textarea>');
        $('body').append($temp);
        $temp.val(text).select();
        document.execCommand('copy');
        $temp.remove();
        
        // Show feedback
        const $tooltip = $('<span class="copy-tooltip">Kopiert!</span>');
        $(this).after($tooltip);
        
        setTimeout(function() {
            $tooltip.fadeOut(function() {
                $tooltip.remove();
            });
        }, 1000);
    });

    // Advanced settings toggle
    let advancedVisible = false;
    
    function toggleAdvancedSettings() {
        const $advanced = $('.advanced-setting');
        
        if (advancedVisible) {
            $advanced.hide();
            $('#toggle-advanced').text('Erweiterte Einstellungen anzeigen');
        } else {
            $advanced.show();
            $('#toggle-advanced').text('Erweiterte Einstellungen ausblenden');
        }
        
        advancedVisible = !advancedVisible;
    }

    // Add advanced toggle button if needed
    if ($('.advanced-setting').length > 0) {
        const $toggleBtn = $('<button type="button" id="toggle-advanced" class="button" style="margin-bottom: 20px;">Erweiterte Einstellungen anzeigen</button>');
        $('.wp-c2pa-admin-section h2').first().after($toggleBtn);
        $toggleBtn.on('click', toggleAdvancedSettings);
        
        // Hide advanced settings initially
        $('.advanced-setting').hide();
    }

    // Interactive help tooltips
    $('[data-tooltip]').each(function() {
        const tooltip = $(this).data('tooltip');
        $(this).attr('title', tooltip);
    });

    // Form dirty checking
    let formIsDirty = false;
    
    $('.badge-setting').on('change input', function() {
        formIsDirty = true;
    });
    
    $(window).on('beforeunload', function(e) {
        if (formIsDirty) {
            const message = 'Sie haben ungespeicherte Änderungen. Möchten Sie die Seite wirklich verlassen?';
            e.returnValue = message;
            return message;
        }
    });
    
    $('#c2pa-settings-form').on('submit', function() {
        formIsDirty = false;
    });

    // Initialize status
    updateStatus();

    // Badge hover effects in preview
    $('#badge-preview').on('mouseenter', function() {
        $(this).css('transform', 'translateY(-2px) scale(1.05)');
    }).on('mouseleave', function() {
        $(this).css('transform', 'translateY(0) scale(1)');
    });

    // Live validation
    $('input[data-setting="badge_text"]').on('input', function() {
        const value = $(this).val();
        const $field = $(this);
        const display = $('select[data-setting="badge_display"]').val() || 'icon_text';
        
        // Skip validation if "icon_only" is selected
        if (display === 'icon_only') {
            $field.css('border-color', '#ddd');
            hideFieldError($field);
            return;
        }
        
        if (value.length > 50) {
            $field.css('border-color', '#e74c3c');
            showFieldError($field, 'Text zu lang (max. 50 Zeichen)');
        } else if (value.length === 0) {
            $field.css('border-color', '#e74c3c');
            showFieldError($field, 'Text ist erforderlich');
        } else {
            $field.css('border-color', '#27ae60');
            hideFieldError($field);
        }
    });

    function showFieldError($field, message) {
        const $error = $field.siblings('.field-error');
        if ($error.length === 0) {
            $field.after('<span class="field-error" style="color: #e74c3c; font-size: 12px; display: block; margin-top: 5px;">' + message + '</span>');
        } else {
            $error.text(message);
        }
    }

    function hideFieldError($field) {
        $field.siblings('.field-error').remove();
    }

    // Add loading states
    function showLoading($element) {
        $element.prop('disabled', true).addClass('loading');
    }

    function hideLoading($element) {
        $element.prop('disabled', false).removeClass('loading');
    }

    // Export/Import settings functionality (for future enhancement)
    function exportSettings() {
        const settings = {};
        $('.badge-setting').each(function() {
            const $field = $(this);
            const setting = $field.data('setting');
            
            if ($field.is(':radio')) {
                if ($field.is(':checked')) {
                    settings[setting] = $field.val();
                }
            } else {
                settings[setting] = $field.val();
            }
        });
        
        return JSON.stringify(settings, null, 2);
    }

    function importSettings(settingsJson) {
        try {
            const settings = JSON.parse(settingsJson);
            
            Object.keys(settings).forEach(function(key) {
                const $field = $('.badge-setting[data-setting="' + key + '"]');
                
                if ($field.is(':radio')) {
                    $field.filter('[value="' + settings[key] + '"]').prop('checked', true);
                } else {
                    $field.val(settings[key]);
                }
            });
            
            updateBadgePreview();
            showNotification('Einstellungen erfolgreich importiert', 'success');
            
        } catch (e) {
            showNotification('Fehler beim Importieren der Einstellungen', 'error');
        }
    }

    // Keyboard shortcuts
    $(document).on('keydown', function(e) {
        if (e.ctrlKey || e.metaKey) {
            switch (e.keyCode) {
                case 83: // Ctrl+S - Save
                    e.preventDefault();
                    $('#c2pa-settings-form').submit();
                    break;
                case 82: // Ctrl+R - Reset
                    e.preventDefault();
                    $('#reset-settings').click();
                    break;
            }
        }
    });

    // Add keyboard shortcut hints
    if ($('.wp-c2pa-admin-actions').length > 0) {
        $('.wp-c2pa-admin-actions').append('<small style="color: #666; margin-left: auto;">Shortcuts: Strg+S (Speichern), Strg+R (Zurücksetzen)</small>');
    }

    console.log('C2PA Admin Panel initialized successfully');
});
