/* WP C2PA Viewer Combined – Vereinheitlichte Version */
(function(){
  'use strict';

  const cfg = {
    version: (typeof WPC2PACombinedConfig!=='undefined'&&WPC2PACombinedConfig.c2paVersion)||'0.30.11',
    settings: (typeof WPC2PACombinedConfig!=='undefined'?WPC2PACombinedConfig:{}),
    c2pa: null,
    ready: false,
    cache: new Map(),
    notificationsShown: new Set(),
    // Globale Badge-Verwaltung
    attachedBadges: new Set(),
    debug: {
      enabled: false,
      errors: [],
      warnings: [],
      info: []
    }
  };

  // Plugin URL für Asset-Pfade
  const pluginUrl = (typeof WPC2PACombinedConfig!=='undefined'&&WPC2PACombinedConfig.pluginUrl)||window.location.origin + '/wp-content/plugins/wp-c2pa-viewer-combined/';

  // Debug-Funktion
  function debug(type, message, data = null) {
    if (!cfg.debug.enabled) return;
    
    const timestamp = new Date().toISOString();
    const logEntry = {
      timestamp,
      type,
      message,
      data,
      pluginVersion: cfg.version,
      themeMode: cfg.settings.themeCompatibilityMode || 'standard'
    };
    
    switch(type) {
      case 'error':
        console.error('🚨 C2PA ERROR:', message, data);
        cfg.debug.errors.push(logEntry);
        break;
      case 'warning':
        console.warn('⚠️ C2PA WARNING:', message, data);
        cfg.debug.warnings.push(logEntry);
        break;
      case 'info':
        console.log('ℹ️ C2PA INFO:', message, data);
        cfg.debug.info.push(logEntry);
        break;
      default:
        console.log('🔍 C2PA DEBUG:', message, data);
        cfg.debug.info.push(logEntry);
    }
  }

  // Badge-ID-Generator für eindeutige Identifikation
  function generateBadgeId(el) {
    const src = el.src || el.dataset.src || '';
    const className = el.className || '';
    const parentId = el.parentElement ? el.parentElement.id || '' : '';
    return `badge_${src.length}_${className.length}_${parentId.length}_${Date.now()}`;
  }

  // NEUE LÖSUNG: Bereinige alle bestehenden Badges
  function cleanupExistingBadges() {
    debug('info', 'Bereinige bestehende Badges');
    
    // Entferne alle Badges aus dem DOM
    const existingBadges = document.querySelectorAll('.wp-c2pa-combined-badge');
    existingBadges.forEach(badge => {
      debug('info', 'Entferne bestehenden Badge', badge);
      badge.remove();
    });
    
    // Entferne alle Badge-Container
    const existingContainers = document.querySelectorAll('.wp-c2pa-combined-badge-container');
    existingContainers.forEach(container => {
      // Verschiebe das Bild zurück an seine ursprüngliche Position
      const img = container.querySelector('img');
      if (img && container.parentNode) {
        container.parentNode.insertBefore(img, container);
        container.remove();
      }
    });
    
    // Lösche alle Badge-Datasets
    const processedImages = document.querySelectorAll('[data-wp-c2pa-combined-processed]');
    processedImages.forEach(img => {
      delete img.dataset.wpC2paCombinedProcessed;
      delete img.dataset.wpC2paCombinedBadgeAttached;
    });
    
    // Lösche globale Badge-Liste
    cfg.attachedBadges.clear();
    
    debug('info', 'Alle bestehenden Badges bereinigt');
  }

  // Debug-Export für Browser-Konsole
  window.C2PADebug = {
    enable: () => {
      cfg.debug.enabled = true;
      console.log('🔧 C2PA Debug aktiviert');
    },
    disable: () => {
      cfg.debug.enabled = false;
      console.log('🔧 C2PA Debug deaktiviert');
    },
    getLogs: () => {
      return {
        errors: cfg.debug.errors,
        warnings: cfg.debug.warnings,
        info: cfg.debug.info
      };
    },
    clearLogs: () => {
      cfg.debug.errors = [];
      cfg.debug.warnings = [];
      cfg.debug.info = [];
      console.log('🧹 C2PA Debug-Logs gelöscht');
    },
    exportLogs: () => {
      const logs = {
        timestamp: new Date().toISOString(),
        pluginVersion: cfg.version,
        themeMode: cfg.settings.themeCompatibilityMode || 'standard',
        errors: cfg.debug.errors,
        warnings: cfg.debug.warnings,
        info: cfg.debug.info
      };
      console.log('📋 C2PA Debug-Export:', logs);
      return logs;
    },
    // Neue Debug-Funktionen für Badge-Verwaltung
    getAttachedBadges: () => {
      return Array.from(cfg.attachedBadges);
    },
    clearAttachedBadges: () => {
      cfg.attachedBadges.clear();
      console.log('🧹 Alle angehängten Badges gelöscht');
    }
  };

  // Translation function
  function t(key) {
    const currentLanguage = cfg.settings.language || 'de';
    
    const translations = {
      de: {
        // Notifications
        'contentCredentialsFound': 'Content Credentials gefunden',
        'metadataFound': 'Metadaten gefunden',
        'noCredentialsFound': 'Keine Content Credentials gefunden',
        'loading': 'Lade Content Credentials...',
        
        // Summary Tab
        'summary': 'Zusammenfassung',
        'noSummaryAvailable': 'Keine Zusammenfassung verfügbar.',
        
        // Summary Fields
        'fileName': 'Name der Datei',
        'issuedBy': 'Ausgestellt von oder Autor',
        'usedApplication': 'Verwendete Applikation',
        'dimensions': 'Abmessungen',
        'editing': 'Bearbeitung',
        'layers': 'Ebenen',
        'description': 'Prompt',
        'creationTime': 'Erstellungszeit',
        'camera': 'Kamera',
        'lens': 'Objektiv',
        
        // Extended View Tab
        'extendedView': 'Erweiterte Ansicht',
        'showDetails': 'Details anzeigen',
        'editingSteps': 'Bearbeitungsschritte',
        'layers': 'Ebenen',
        
        // AI Indicators
        'aiImage': 'KI Bild',
        'midjourney': 'Midjourney',
        'photoshop': 'Photoshop',
        'sora': 'Sora',
        'openai': 'OpenAI',
        'chatgpt': 'ChatGPT',
        'canva': 'Canva',
        'adobeFirefly': 'Adobe Firefly'
      },
      en: {
        // Notifications
        'contentCredentialsFound': 'Content Credentials found',
        'metadataFound': 'Metadata found',
        'noCredentialsFound': 'No Content Credentials found',
        'loading': 'Loading Content Credentials...',
        
        // Summary Tab
        'summary': 'Summary',
        'noSummaryAvailable': 'No summary available.',
        
        // Summary Fields
        'fileName': 'File Name',
        'issuedBy': 'Issued by or Author',
        'usedApplication': 'Used Application',
        'dimensions': 'Dimensions',
        'editing': 'Editing',
        'layers': 'Layers',
        'description': 'Prompt',
        'creationTime': 'Creation Time',
        'camera': 'Camera',
        'lens': 'Lens',
        
        // Extended View Tab
        'extendedView': 'Extended View',
        'showDetails': 'Show Details',
        'editingSteps': 'Editing Steps',
        'layers': 'Layers',
        
        // AI Indicators
        'aiImage': 'AI Image',
        'midjourney': 'Midjourney',
        'photoshop': 'Photoshop',
        'sora': 'Sora',
        'openai': 'OpenAI',
        'chatgpt': 'ChatGPT',
        'canva': 'Canva',
        'adobeFirefly': 'Adobe Firefly'
      }
    };
    
    return translations[currentLanguage][key] || key;
  }

  function notify(msg,type='info',ms=2000){
    // Prüfen ob diese Benachrichtigung bereits angezeigt wurde
    const notificationKey = `${msg}-${type}`;
    if(cfg.notificationsShown.has(notificationKey)) return;
    cfg.notificationsShown.add(notificationKey);
    
    const n=document.createElement('div');
    n.className=`wp-c2pa-combined-notification wp-c2pa-combined-${type}`; n.textContent=msg;
    document.body.appendChild(n); requestAnimationFrame(()=>{ n.classList.add('wp-c2pa-combined-slide-in'); });
    setTimeout(()=>{ n.classList.add('wp-c2pa-combined-slide-out'); setTimeout(()=>n.remove(),300); },ms);
  }

  async function loadC2PA(){
    try{
      const url=`https://cdn.jsdelivr.net/npm/c2pa@${cfg.version}/+esm`;
      const { createC2pa } = await import(/* @vite-ignore */ url);
      cfg.c2pa = await createC2pa({
        wasmSrc:`https://cdn.jsdelivr.net/npm/c2pa@${cfg.version}/dist/assets/wasm/toolkit_bg.wasm`,
        workerSrc:`https://cdn.jsdelivr.net/npm/c2pa@${cfg.version}/dist/c2pa.worker.min.js`,
      });
      cfg.ready=true;
    }catch(e){ cfg.ready=false; }
  }

  // exifr Loader
  let exifrReady=false; 
  async function loadExifr(){ 
    if(exifrReady) return window.exifr; 
    return new Promise((res,rej)=>{ 
      const s=document.createElement('script'); 
      s.src='https://cdn.jsdelivr.net/npm/exifr/dist/full.umd.js'; 
      s.async=true; 
      s.onload=()=>{exifrReady=true; res(window.exifr)}; 
      s.onerror=()=>rej(new Error('exifr load fail')); 
      document.head.appendChild(s); 
    }); 
  }

  function parseSrcset(srcset){ 
    if(!srcset) return []; 
    return srcset.split(',').map(p=>{ 
      const m=p.trim().match(/^(\S+)(?:\s+(\d+)w)?/); 
      return m?{url:m[1],w:m[2]?parseInt(m[2],10):0}:null; 
    }).filter(Boolean); 
  }
  
  function findLinkedOriginal(img){ 
    let el=img; 
    for(let i=0;i<3&&el;i++){ 
      const p=el.parentElement; 
      if(!p) break; 
      if(p.tagName==='A'){ 
        const href=p.getAttribute('href')||''; 
        if(/\.(jpe?g|png|webp|gif|tiff?|bmp|heic|heif|svg)(\?.*)?$/i.test(href)) return href; 
      } 
      el=p; 
    } 
    return null; 
  }
  
  function isUIElement(el) {
    // Prüfe CSS-Klassen die auf UI-Elemente hinweisen
    const uiClasses = [
      'logo', 'icon', 'favicon', 'site-icon', 'header-logo', 'footer-logo',
      'menu-icon', 'hamburger', 'close-icon', 'search-icon', 'social-icon',
      'avatar', 'profile-pic', 'user-avatar', 'admin-avatar',
      'button-icon', 'nav-icon', 'breadcrumb-icon', 'pagination-icon',
      'wp-block-site-logo', 'custom-logo', 'site-branding',
      'widget-icon', 'sidebar-icon', 'footer-icon', 'header-icon',
      // Elementor-spezifische UI-Elemente
      'elementor-icon', 'elementor-logo', 'elementor-button-icon',
      'elementor-nav-icon', 'elementor-social-icon', 'elementor-avatar'
    ];
    
    // Prüfe CSS-Klassen
    if (el.className) {
      const classNames = el.className.toLowerCase().split(' ');
      for (const uiClass of uiClasses) {
        if (classNames.includes(uiClass)) {
          return true;
        }
      }
    }
    
    // Prüfe ID
    if (el.id) {
      const id = el.id.toLowerCase();
      for (const uiClass of uiClasses) {
        if (id.includes(uiClass)) {
          return true;
        }
      }
    }
    
    // Prüfe src-Attribut für bekannte UI-Bilder
    if (el.src) {
      const src = el.src.toLowerCase();
      const uiPatterns = [
        'favicon', 'icon', 'logo', 'avatar', 'site-icon',
        'wp-content/uploads/site-icon', 'wp-content/uploads/logo',
        'wp-content/uploads/icon', 'wp-content/uploads/avatar'
      ];
      
      for (const pattern of uiPatterns) {
        if (src.includes(pattern)) {
          return true;
        }
      }
    }
    
    // Prüfe Größe (sehr kleine Bilder sind oft Icons)
    if (el.naturalWidth && el.naturalHeight) {
      if (el.naturalWidth <= 64 && el.naturalHeight <= 64) {
        return true;
      }
    }
    
    // Prüfe alt-Attribut
    if (el.alt) {
      const alt = el.alt.toLowerCase();
      const uiAltPatterns = ['logo', 'icon', 'favicon', 'avatar', 'site icon'];
      for (const pattern of uiAltPatterns) {
        if (alt.includes(pattern)) {
          return true;
        }
      }
    }
    
    return false;
  }
  

  
  function isElementorImageFormat(el) {
    // Prüfe ob das Bild in einem Elementor-Kontext ist
    const elementorSelectors = [
      '.elementor-widget-image',
      '.elementor-image',
      '.elementor-widget-container',
      '.elementor-section',
      '.elementor-column',
      '.elementor-widget',
      '[data-elementor-type]',
      '[data-elementor-id]'
    ];
    
    // Prüfe ob das Bild oder ein Elternelement Elementor-Klassen hat
    for (const selector of elementorSelectors) {
      if (el.closest(selector)) {
        return true;
      }
    }
    
    // Prüfe ob das Bild selbst Elementor-Klassen hat
    if (el.className && el.className.toLowerCase().includes('elementor')) {
      return true;
    }
    
    // Prüfe ob ein Elternelement Elementor-Klassen hat
    let parent = el.parentElement;
    while (parent) {
      if (parent.className && parent.className.toLowerCase().includes('elementor')) {
        return true;
      }
      parent = parent.parentElement;
    }
    
    return false;
  }
  
  function isHomepage() {
    // Prüfe verschiedene Indikatoren für die Startseite
    const currentUrl = window.location.href;
    const currentPath = window.location.pathname;
    
    // WordPress-spezifische Startseite-Erkennung
    if (currentPath === '/' || currentPath === '/index.php') {
      return true;
    }
    
    // Prüfe ob es die Blog-Startseite ist
    if (currentPath.includes('/blog') && (currentPath === '/blog' || currentPath === '/blog/')) {
      return true;
    }
    
    // Prüfe WordPress-Body-Klassen
    const bodyClasses = document.body.className.toLowerCase();
    const homepageClasses = ['home', 'homepage', 'front-page', 'blog'];
    for (const className of homepageClasses) {
      if (bodyClasses.includes(className)) {
        return true;
      }
    }
    
    // Prüfe WordPress-Template
    const template = document.querySelector('meta[name="template"]');
    if (template && template.content) {
      const templateContent = template.content.toLowerCase();
      if (templateContent.includes('home') || templateContent.includes('front-page')) {
        return true;
      }
    }
    
    // Prüfe WordPress-Title
    const title = document.title.toLowerCase();
    if (title.includes('home') || title.includes('startseite') || title.includes('blog')) {
      return true;
    }
    
    return false;
  }

  function isFeaturedImage(el) {
    // Erweiterte Liste von CSS-Klassen die auf Titelbilder hinweisen
    const featuredClasses = [
      'post-thumbnail', 'featured-image', 'entry-thumbnail', 'post-image',
      'wp-post-image', 'attachment-post-thumbnail', 'attachment-featured',
      'hero-image', 'banner-image', 'cover-image', 'header-image',
      'post-thumb', 'entry-image', 'article-image', 'content-image',
      'main-image', 'primary-image', 'lead-image', 'story-image',
      'blog-image', 'page-image', 'single-image', 'archive-image'
    ];
    
    // Prüfe CSS-Klassen des Bildes selbst
    if (el.className) {
      const classNames = el.className.toLowerCase().split(' ');
      for (const featuredClass of featuredClasses) {
        if (classNames.includes(featuredClass)) {
          return true;
        }
      }
    }
    
    // Prüfe ob das Bild in einem Titelbild-Container ist (bis 5 Ebenen hoch)
    let parent = el.parentElement;
    for (let i = 0; i < 5 && parent; i++) {
      if (parent.className) {
        const parentClasses = parent.className.toLowerCase().split(' ');
        for (const featuredClass of featuredClasses) {
          if (parentClasses.includes(featuredClass)) {
            return true;
          }
        }
      }
      
      // Prüfe auch ID des Parents
      if (parent.id) {
        const parentId = parent.id.toLowerCase();
        const idPatterns = ['featured', 'thumbnail', 'hero', 'banner', 'cover', 'header'];
        for (const pattern of idPatterns) {
          if (parentId.includes(pattern)) {
            return true;
          }
        }
      }
      
      parent = parent.parentElement;
    }
    
    // Prüfe WordPress-spezifische Titelbild-Selektoren
    if (el.closest('.post-thumbnail, .featured-image, .entry-thumbnail, .wp-post-image, .post-thumb, .entry-image, .article-image, .content-image, .main-image, .primary-image, .lead-image, .story-image, .blog-image, .page-image, .single-image, .archive-image')) {
      return true;
    }
    
    // Prüfe ob das Bild das erste Bild im Content-Bereich ist
    const contentArea = el.closest('article, .entry, .post, .content, .main-content, .post-content, .entry-content');
    if (contentArea) {
      const allImages = contentArea.querySelectorAll('img');
      if (allImages.length > 0 && allImages[0] === el) {
        // Prüfe ob es wahrscheinlich ein Titelbild ist (größer als andere Bilder)
        let isLargest = true;
        for (let i = 1; i < allImages.length; i++) {
          if (allImages[i].naturalWidth > el.naturalWidth || allImages[i].naturalHeight > el.naturalHeight) {
            isLargest = false;
            break;
          }
        }
        if (isLargest && el.naturalWidth > 300 && el.naturalHeight > 200) {
          return true;
        }
      }
    }
    
    // Prüfe alt-Attribut für Titelbild-Hinweise
    if (el.alt) {
      const alt = el.alt.toLowerCase();
      const altPatterns = ['featured', 'thumbnail', 'hero', 'banner', 'cover', 'header', 'main image', 'lead image'];
      for (const pattern of altPatterns) {
        if (alt.includes(pattern)) {
          return true;
        }
      }
    }
    
    return false;
  }

  function chooseBestUrl(el){ 
    const exp=el.getAttribute('data-full-url')||el.getAttribute('data-orig-src')||el.getAttribute('data-original'); 
    if(exp) return exp; 
    if(el.tagName==='VIDEO'){ 
      const v=el; 
      if(v.currentSrc) return v.currentSrc; 
      if(v.src) return v.currentSrc; 
      const s=v.querySelector&&v.querySelector('source[src]'); 
      if(s) return s.getAttribute('src'); 
    }
    const linked=findLinkedOriginal(el); 
    if(linked) return linked; 
    const c=parseSrcset(el.getAttribute('srcset')); 
    if(c.length){ 
      c.sort((a,b)=>(b.w||0)-(a.w||0)); 
      return c[0].url||el.currentSrc||el.src; 
    } 
    return el.currentSrc||el.src; 
  }
  
  function parseLinkHeader(h){ 
    if(!h) return []; 
    return h.split(',').map(p=>p.trim()).map(i=>{ 
      const m=i.match(/^<([^>]+)>\s*;\s*rel="?([^";]+)"?/i); 
      return m?{url:m[1],rel:m[2].toLowerCase()}:null; 
    }).filter(Boolean); 
  }
  
  function toAbs(base,u){ 
    try{ return new URL(u,base).toString(); }catch{ return u; } 
  }

  async function readManifest(el){ 
    if(!cfg.ready||!cfg.c2pa) return null; 
    const src=chooseBestUrl(el); 
    if(!src) return null; 
    if(cfg.cache.has(src)) return cfg.cache.get(src);
    try{
      let r=await fetch(src,{mode:'cors'}); 
      if(!r.ok){ 
        const fb=el.currentSrc||el.src; 
        if(fb&&fb!==src) r=await fetch(fb,{mode:'cors'}); 
      }
      if(!r.ok) throw new Error('HTTP '+r.status);
      const blob=await r.blob(); 
      const res=await cfg.c2pa.read(blob);
      if(res&&res.manifestStore&&res.manifestStore.activeManifest){ 
        const a=res.manifestStore.activeManifest; 
        const m={ 
          label:a.label,title:a.title,format:a.format,claimGenerator:a.claimGenerator,claimGeneratorInfo:a.claimGeneratorInfo||(a.claim&&a.claim.claim_generator_info)||[], 
          signatureInfo:a.signatureInfo,claim:a.claim||{}, assertions:a.assertions||{}, ingredients:a.ingredients||[], actions:[] 
        };
        // Helper to fetch assertion by label from object or array forms
        const getByLabel=(as,label)=>{ 
          if(!as) return null; 
          if(Array.isArray(as)){ 
            const low=label.toLowerCase(); 
            return as.find(it=>{ 
              const l=(it&& (it.label||it.type||''))||''; 
              return l.toLowerCase()===low; 
            })||null; 
          } 
          return as[label]||null; 
        };
        
        // Parse assertions - handle both direct object and data array format
        if(a.assertions){
          let assertionsToCheck = a.assertions;
          
          // Check if assertions is in data array format
          if(a.assertions.data && Array.isArray(a.assertions.data)){
            assertionsToCheck = a.assertions.data;
          }
          
          const a1 = getByLabel(assertionsToCheck,'c2pa.actions');
          if(a1){ 
            const list=(a1.actions)||(a1.data&&a1.data.actions); 
            if(Array.isArray(list)) m.actions = list.slice(); 
          }
          const a2 = getByLabel(assertionsToCheck,'c2pa.actions.v2') || getByLabel(assertionsToCheck,'c2pa.actionsV2');
          if(a2){ 
            const list=(a2.data&&a2.data.actions)||a2.actions; 
            if(Array.isArray(list)) m.actions = (m.actions||[]).concat(list); 
            const meta=(a2.data&&a2.data.metadata)||a2.metadata; 
            if(meta){ m.actionsMeta = meta; } 
          }
          
          // Creator from schema.org CreativeWork
          const cw = getByLabel(assertionsToCheck,'stds.schema-org.CreativeWork') || getByLabel(assertionsToCheck,'schema-org.CreativeWork');
          try{ 
            const data = cw && (cw.data||cw.value||cw.assertion||cw);
            const creators = data && (data.author||data.creator||data.name||data.attributedTo||data['@list']||data['@graph']);
            const arr = Array.isArray(creators)?creators:(creators?[creators]:[]);
            const first = arr[0]||{}; 
            const name = first.name||first['schema:name']||first.label||null; 
            const identifier = first.identifier||first['schema:identifier']||null; 
            if(name||identifier) m.creator={ name, identifier }; 
          }catch{}
        }
        cfg.cache.set(src,m); 
        return m; 
      }
      // decoupled
      const links=parseLinkHeader(r.headers.get('Link')||r.headers.get('link')); 
      const lk=links.find(l=>l.rel==='c2pa'||l.rel==='c2pa.manifest');
      if(lk){ 
        const mu=toAbs(r.url,lk.url); 
        const mr=await fetch(mu,{mode:'cors'}); 
        if(mr.ok){ 
          const j=await mr.json(); 
          const a=(j&&j.manifestStore&&j.manifestStore.activeManifest)||j.activeManifest||j; 
          if(a){ 
            const m={ 
              label:a.label,title:a.title,format:a.format,claimGenerator:a.claimGenerator,claimGeneratorInfo:a.claimGeneratorInfo||(a.claim&&a.claim.claim_generator_info)||[], 
              signatureInfo:a.signatureInfo,claim:a.claim||{}, assertions:a.assertions||{}, ingredients:a.ingredients||[], actions:[], _source:'decoupled' 
            };
            const getByLabel=(as,label)=>{ 
              if(!as) return null; 
              if(Array.isArray(as)){ 
                const low=label.toLowerCase(); 
                return as.find(it=>{ 
                  const l=(it&&(it.label||it.type||''))||''; 
                  return l.toLowerCase()===low; 
                })||null; 
              } 
              return as[label]||null; 
            };
            
            // Parse assertions - handle both direct object and data array format
            if(a.assertions){
              let assertionsToCheck = a.assertions;
              
              // Check if assertions is in data array format
              if(a.assertions.data && Array.isArray(a.assertions.data)){
                assertionsToCheck = a.assertions.data;
              }
              
              const a1 = getByLabel(assertionsToCheck,'c2pa.actions');
              if(a1){ 
                const list=(a1.actions)||(a1.data&&a1.data.actions); 
                if(Array.isArray(list)) m.actions = list.slice(); 
              }
              const a2 = getByLabel(assertionsToCheck,'c2pa.actions.v2') || getByLabel(assertionsToCheck,'c2pa.actionsV2');
              if(a2){ 
                const list=(a2.data&&a2.data.actions)||a2.actions; 
                if(Array.isArray(list)) m.actions = (m.actions||[]).concat(list); 
                const meta=(a2.data&&a2.data.metadata)||a2.metadata; 
                if(meta){ m.actionsMeta = meta; } 
              }
            }
            cfg.cache.set(src,m); 
            return m; 
          } 
        } 
      }
      return null;
    }catch{ return null; }
  }

  async function readMetadata(el){ 
    const src=chooseBestUrl(el); 
    try{ 
      const exifr=await loadExifr(); 
      if(!src) return {raw:{},summary:{}}; 
      const r=await fetch(src,{mode:'cors'}); 
      let meta=null, blob=null; 
      if(r&&r.ok){ 
        blob=await r.blob(); 
        try{ 
          meta=await exifr.parse(blob,{tiff:true,exif:true,iptc:true,xmp:true,jfif:true}); 
        }catch{} 
      }
      return { raw: meta||{}, summary: summarize(meta, blob, el, src) };
    }catch{ return {raw:{},summary:{}}; } 
  }
  
  function summarize(meta, blob, el, src){ 
    if(!meta) return {}; 
    const created=meta.CreateDate||meta.DateTimeOriginal||meta.DateCreated||meta.XPCreateDate||meta.ModifyDate||meta['xmp:CreateDate']; 
    const modified=meta.ModifyDate||meta['xmp:ModifyDate']||meta['photoshop:DateCreated']; 
    const software=meta.Software||meta.CreatorTool||meta['xmp:CreatorTool']||meta['tiff:Software']; 
    const artist=meta.Artist||meta.Author||meta.Creator||meta['dc:creator']; 
    const camera=meta.Model||meta.CameraModelName; 
    const make=meta.Make; 
    let w=meta.ImageWidth||meta.PixelXDimension||meta.ExifImageWidth; 
    let h=meta.ImageHeight||meta.PixelYDimension||meta.ExifImageHeight; 
    if((!w||!h)&&el){ w=el.naturalWidth||el.width||w; h=el.naturalHeight||el.height||h; } 
    return { 
      createdAt:created||null, modifiedAt:modified||null, software:software||null, artist:artist||null, 
      cameraMake:make||null, cameraModel:camera||null, dimensions:(w&&h)?`${w}×${h}`:null, 
      mimeType:(blob&&blob.type)||(src&&String(src).split('?')[0].split('.').pop())||null 
    } 
  }
  
  function formatDate(s){ 
    try{return s?new Date(s).toLocaleString('de-DE'):'—'}catch{return s||'—'} 
  }
  
  function getActionDescription(a){ 
    const m={ 
      'c2pa.opened':'Datei geöffnet','c2pa.placed':'Inhalt importiert','c2pa.edited':'Bearbeitet','c2pa.drawing':'Zeichnung/Malen',
      'c2pa.color_adjustments':'Farb-/Belichtungsanpassung','c2pa.filter':'Filter angewendet','c2pa.crop':'Zugeschnitten',
      'c2pa.resize':'Größe geändert','c2pa.relight':'Beleuchtung angepasst','c2pa.unknown':'Unbekannte Bearbeitung',
      'c2pa.removed':'Inhalt entfernt','c2pa.added':'Inhalt hinzugefügt','c2pa.transformed':'Transformiert',
      'c2pa.masked':'Maskiert','c2pa.cloned':'Geklont/Kopiert','c2pa.healing':'Repariert/Geheilt'
    }; 
    return m[a]||a||''; 
  }
  
  function getDetailedActionDescription(action, params){ 
    const actionDescriptions = {
      'c2pa.opened': 'Datei wurde geöffnet',
      'c2pa.placed': 'Vorhandener Inhalt wurde in die Datei eingefügt',
      'c2pa.edited': 'Sonstige Bearbeitungen wurden vorgenommen',
      'c2pa.drawing': 'Zeichenwerkzeuge wurden verwendet (Pinsel, Stift, etc.)',
      'c2pa.color_adjustments': 'Farb- oder Belichtungseinstellungen wurden angepasst',
      'c2pa.filter': 'Filter oder Effekte wurden angewendet',
      'c2pa.crop': 'Bild wurde zugeschnitten',
      'c2pa.resize': 'Bildgröße wurde verändert',
      'c2pa.relight': 'Beleuchtung oder Schatten wurden angepasst',
      'c2pa.unknown': 'Unbekannte oder nicht erkannte Bearbeitung',
      'c2pa.removed': 'Inhalte wurden entfernt oder gelöscht',
      'c2pa.added': 'Neue Inhalte wurden hinzugefügt',
      'c2pa.transformed': 'Geometrische Transformationen (Drehen, Skalieren, etc.)',
      'c2pa.masked': 'Masken wurden angewendet oder bearbeitet',
      'c2pa.cloned': 'Bereiche wurden geklont oder dupliziert',
      'c2pa.healing': 'Heilungs- oder Reparaturwerkzeuge wurden verwendet'
    };
    
    let description = actionDescriptions[action] || action || 'Unbekannte Aktion';
    
    // Erweitere Beschreibung basierend auf Parametern
    if (params) {
      if (params.description) {
        description += ` (${params.description})`;
      }
      
      // Spezielle Behandlung für bestimmte Parameter
      if (params['com.adobe.tool']) {
        const toolName = getPhotoshopToolName(params['com.adobe.tool']);
        description += ` - Tool: ${toolName}`;
      }
      
      if (params.name) {
        const toolName = getPhotoshopToolName(params.name);
        description += ` - ${toolName}`;
      }
    }
    
    return description;
  }
  
  function getPhotoshopToolName(toolKey) {
    const toolNames = {
      'open': 'Datei öffnen',
      'neural_filters': 'Neuronale Filter',
      'paste': 'Einfügen',
      'move': 'Verschieben-Werkzeug',
      'free_transform': 'Frei transformieren',
      'add_layer_mask': 'Ebenenmaske hinzufügen',
      'brush_tool': 'Pinsel-Werkzeug',
      'gaussian_blur': 'Gaußscher Weichzeichner',
      'spot_healing_brush': 'Bereichsreparaturpinsel',
      'layer_order': 'Ebenenreihenfolge ändern',
      'layer_visibility': 'Ebenensichtbarkeit',
      'black__white': 'Schwarzweiß-Anpassung',
      'img_8348.heic': 'Bild importiert',
      'curves': 'Gradationskurven',
      'levels': 'Tonwertkorrektur',
      'hue_saturation': 'Farbton/Sättigung',
      'color_balance': 'Farbbalance',
      'brightness_contrast': 'Helligkeit/Kontrast',
      'vibrance': 'Dynamik',
      'clone_stamp': 'Kopierstempel',
      'healing_brush': 'Reparaturpinsel',
      'patch_tool': 'Ausbessern-Werkzeug',
      'eraser': 'Radiergummi',
      'magic_wand': 'Zauberstab',
      'lasso': 'Lasso-Werkzeug',
      'marquee': 'Auswahlrechteck'
    };
    
    return toolNames[toolKey] || toolKey || 'Unbekanntes Werkzeug';
  }
  
  function safeJSON(o,sp){ 
    const seen=new WeakSet(); 
    return JSON.stringify(o,(k,v)=>{ 
      if(v&&typeof v==='object'){ 
        if(seen.has(v)) return '[Circular]'; 
        seen.add(v);
      } 
      return v; 
    },sp); 
  }

  function renderBasic(man, meta, src, el){ 
    if(!man) return '<div class="wp-c2pa-combined-box-sub">Keine C2PA-Daten</div>';
    const sig=man.signatureInfo||{}, claim=man.claim||{}, ginf=man.claimGeneratorInfo||(claim&&claim.claim_generator_info)||[], gen=ginf&&ginf[0]||null, ings=man.ingredients||[], assertions=man.assertions||{};
    // Dateiname aus URL ermitteln
    let fileName='—'; 
    try{ 
      if(src){ 
        const u=new URL(src, window.location.href); 
        fileName=decodeURIComponent(u.pathname.split('/').pop()||'—'); 
      } 
    }catch{}
    // Abmessungen aus Metadaten oder Element ableiten
    let dims=(meta&&meta.summary&&meta.summary.dimensions)||null; 
    if(!dims&&el){ 
      const w=el.naturalWidth||el.width; 
      const h=el.naturalHeight||el.height; 
      if(w&&h) dims=`${w}×${h}`; 
    }
    // Bestandteile zusammenzählen (Ebenen)
    const count = Array.isArray(ings)?ings.length:0;
    const layerText = (count===0)?'keine Ebene':(count===1)?'eine Ebene':(count===2)?'zwei Ebenen':`${count} Ebenen`;
    let html=`<div class="wp-c2pa-combined-grid">
              <div><div class="wp-c2pa-combined-field">${t('fileName')}</div><div class="wp-c2pa-combined-value">${fileName}</div></div>
              <div><div class="wp-c2pa-combined-field">${t('issuedBy')}</div><div class="wp-c2pa-combined-value">${claim.issuer||sig.issuer||'—'}</div></div>
        <div><div class="wp-c2pa-combined-field">${t('creationTime')}</div><div class="wp-c2pa-combined-value">${sig.time?formatDate(sig.time):'—'}</div></div>
              <div><div class="wp-c2pa-combined-field">${t('usedApplication')}</div><div class="wp-c2pa-combined-value">${(gen&&gen.name)||claim.generator||man.claimGenerator||'—'}</div>${(gen&&gen.version)?`<div class=\"wp-c2pa-combined-sub\">Version: ${gen.version}</div>`:''}</div>
        <div><div class="wp-c2pa-combined-field">${t('dimensions')}</div><div class="wp-c2pa-combined-value">${dims||'—'}</div></div>
    </div>`;
    
    // Bestandteile nur als Zähler + aufklappbare Aufschlüsselung
    html+=`<div class="wp-c2pa-combined-section"><div class="wp-c2pa-combined-section-title">Bestandteile</div>
      <div class="wp-c2pa-combined-box">
        <div class="wp-c2pa-combined-box-title">${layerText}</div>
        <details><summary>Aufschlüsselung anzeigen</summary>
          <div class="wp-c2pa-combined-stack">${(ings||[]).map((ing,i)=>{ 
            const label=ing.title||`Bestandteil ${i+1}`; 
            const rel=ing.relationship?`Beziehung: ${ing.relationship}`:''; 
            const fmt=ing.format?`Format: ${ing.format}`:''; 
            const id=ing.instanceId?`ID: ${String(ing.instanceId).slice(0,8)}...`:''; 
            const val=Array.isArray(ing.validationStatus)?ing.validationStatus.map(v=>v.code||v.explanation).filter(Boolean).join(' • '):''; 
            const cred=(ing.relationship==='parentOf'||ing.instanceId||ing.title)?'Hat Content Credential':'Kein Content Credential'; 
            return `<div class=\"wp-c2pa-combined-box\"><div class=\"wp-c2pa-combined-box-title\">${label}</div><div class=\"wp-c2pa-combined-box-sub\">${[cred,fmt,rel,id,val].filter(Boolean).join(' • ')||'—'}</div></div>`; 
          }).join('')}
          </div>
        </details>
      </div>
    </div>`;

    // Änderungen aus Actions (inkl. v2) + Meta (reviewRatings, dateTime)
    const actions = Array.isArray(man.actions)?man.actions:[];
    const actionsMeta = man.actionsMeta || {};
    // Immer anzeigen, auch wenn leer, damit User sieht dass geprüft wurde
    const listHtml = actions.map(a=>{
      const desc=(a.parameters&&a.parameters.description)||''; 
      const tool=(a.parameters&&((a.parameters['com.adobe.tool'])||(a.parameters.name)))||a.softwareAgent||'';
      const iid=(a.parameters&&a.parameters.instanceId)||'';
      return `<div class=\"wp-c2pa-combined-box\"><div class=\"wp-c2pa-combined-box-title\">${getActionDescription(a.action||'')}</div>${tool?`<div class=\"wp-c2pa-combined-box-sub\">Tool: ${tool}</div>`:''}${iid?`<div class=\"wp-c2pa-combined-box-sub\">Instance: ${iid}</div>`:''}${desc?`<div class=\"wp-c2pa-combined-box-sub\">${desc}</div>`:''}</div>`;
    }).join('');
    const ratings = (actionsMeta.reviewRatings||[]).map(r=>`<div class=\"wp-c2pa-combined-box-sub\"><code>${r.code}</code>: ${r.value}${r.explanation?` – ${r.explanation}`:''}</div>`).join('');
    const when = actionsMeta.dateTime?`<div class=\"wp-c2pa-combined-box-sub\">Zeitpunkt: ${formatDate(actionsMeta.dateTime)}</div>`:'';
    html+=`<div class="wp-c2pa-combined-section"><div class="wp-c2pa-combined-section-title">Änderungen</div><div class="wp-c2pa-combined-stack">${listHtml||'<div class=\"wp-c2pa-combined-box-sub\">Keine Änderungen im Manifest aufgeführt.</div>'}${(ratings||when)?`<div class=\"wp-c2pa-combined-box\"><div class=\"wp-c2pa-combined-box-title\">Zusammenfassung</div>${when}${ratings}</div>`:''}</div></div>`;

    // Creator-Block, falls vorhanden (nutze ggf. vorab ermittelte m.creator)
    let creatorName=null, creatorId=null;
    if(!creatorName && man.creator && (man.creator.name||man.creator.identifier)){
      creatorName = man.creator.name||null; 
      creatorId = man.creator.identifier||null;
    }
    if(creatorName){ 
      html+=`<div class="wp-c2pa-combined-section"><div class="wp-c2pa-combined-section-title">Ersteller</div><div class="wp-c2pa-combined-box"><div class="wp-c2pa-combined-box-sub">Name: ${creatorName}</div></div></div>`; 
    }
    return html;
  }

  function renderSummary(man, meta, src, el){ 
    const s=(meta&&meta.summary)||{}; 
    let html='';
    
    // KI-Indikator prüfen (an erster Stelle)
    try{ 
      const rawDataString = safeJSON({manifest:man, metadata:meta},0) || '';
                  const hasIptc = rawDataString.toLowerCase().includes('iptc.org');
            const hasMidjourneyVersion = rawDataString.match(/--v\s+[\d.]+/);
            const hasMidjourney = rawDataString.toLowerCase().includes('midjourney');
                  const hasOpenAI = rawDataString.toLowerCase().includes('openai');
            const hasChatGPT = rawDataString.toLowerCase().includes('chatgpt');
            const hasSora = rawDataString.toLowerCase().includes('sora');
            const hasAdobeFirefly = rawDataString.toLowerCase().includes('adobe_firefly');
            const hasCanva = rawDataString.toLowerCase().includes('canva');
            const hasAdobePhotoshop = (rawDataString.toLowerCase().includes('adobe photoshop') || rawDataString.toLowerCase().includes('adobe_photoshop')) && !rawDataString.toLowerCase().includes('lightroom');
            const hasAdobeLightroom = rawDataString.toLowerCase().includes('lightroom');
            const hasGeminiFlash = rawDataString.toLowerCase().includes('gemini-flash');
            const hasNanoBanana = rawDataString.toLowerCase().includes('nano-banana');
      
              if(hasIptc || hasMidjourneyVersion || hasMidjourney || hasOpenAI || hasChatGPT || hasSora || hasAdobeFirefly || hasCanva || hasAdobePhotoshop || hasAdobeLightroom || hasGeminiFlash || hasNanoBanana){
        html+=`<div class="wp-c2pa-ai-indicator">`;
        if(hasIptc){
          html+=`<div class="wp-c2pa-ai-badge">${t('aiImage')}</div>`;
        }
                      if(hasMidjourneyVersion){
                html+=`<div class="wp-c2pa-midjourney-badge">Midjourney</div>`;
              }
              if(hasMidjourney){
                html+=`<div class="wp-c2pa-midjourney-badge">Midjourney</div>`;
              }
        if(hasOpenAI){
          html+=`<div class="wp-c2pa-openai-badge">OpenAI</div>`;
        }
        if(hasChatGPT){
          html+=`<div class="wp-c2pa-chatgpt-badge">ChatGPT</div>`;
        }
                      if(hasSora){
                html+=`<div class="wp-c2pa-sora-badge">Sora</div>`;
              }
              if(hasAdobeFirefly){
                html+=`<div class="wp-c2pa-adobe-firefly-badge">Adobe Firefly</div>`;
              }
              if(hasCanva){
                html+=`<div class="wp-c2pa-canva-badge">Canva</div>`;
              }
              if(hasAdobePhotoshop){
                html+=`<div class="wp-c2pa-adobe-photoshop-badge">Adobe Photoshop</div>`;
              }
              if(hasAdobeLightroom){
                html+=`<div class="wp-c2pa-adobe-lightroom-badge">Adobe Lightroom</div>`;
              }
              if(hasGeminiFlash){
                html+=`<div class="wp-c2pa-gemini-flash-badge">gemini-flash</div>`;
              }
              if(hasNanoBanana){
                html+=`<div class="wp-c2pa-nano-banana-badge">nano-banana</div>`;
              }
        html+=`</div>`;
      }
    }catch{}
    
    // Zusammenfassung der Informationen
    html += '<div class="wp-c2pa-combined-section">';
    html += '<div class="wp-c2pa-combined-section-title">' + t('summary') + '</div>';
    html += '<div class="wp-c2pa-combined-grid">';
    
    // Name der Datei (aus allen verfügbaren Quellen)
    let fileName = '—';
    if (man && man.title) {
      fileName = man.title;
    } else if (meta && meta.summary && meta.summary.fileName) {
      fileName = meta.summary.fileName;
    } else if (src) {
      try {
        const u = new URL(src, window.location.href);
        fileName = decodeURIComponent(u.pathname.split('/').pop() || '—');
      } catch (e) {
        fileName = src.split('/').pop() || '—';
      }
    }
    html += '<div class="wp-c2pa-combined-field">' + t('fileName') + '</div>';
    html += '<div class="wp-c2pa-combined-value">' + fileName + '</div>';
    
    // Ausgestellt von oder Autor (aus allen verfügbaren Quellen)
    let author = '—';
    if (man && man.creator && man.creator.name) {
      author = man.creator.name;
    } else if (s.artist) {
      author = s.artist;
    } else if (meta && meta.raw && meta.raw.Author) {
      author = meta.raw.Author;
    }
    if (author !== '—') {
      html += '<div class="wp-c2pa-combined-field">' + t('issuedBy') + '</div>';
      html += '<div class="wp-c2pa-combined-value">' + author + '</div>';
    }
    
    // Verwendete Applikation (aus allen verfügbaren Quellen)
    let appDisplay = '—';
    if (man && man.claimGenerator) {
      appDisplay = man.claimGenerator;
      // Kürze Photoshop-Anzeige
      if (appDisplay.toLowerCase().includes('adobe_photoshop') || appDisplay.toLowerCase().includes('adobe photoshop')) {
        const versionMatch = appDisplay.match(/(\d+\.\d+\.\d+)/);
        if (versionMatch) {
          appDisplay = 'Adobe Photoshop ' + versionMatch[1];
        } else {
          appDisplay = appDisplay.replace(/\s+adobe_c2pa\/[^\s]+/g, '').replace(/\s+c2pa-rs\/[^\s]+/g, '');
          appDisplay = appDisplay.replace('Adobe_Photoshop', 'Adobe Photoshop');
        }
      }
      // Kürze Canva-Anzeige
      else if (appDisplay.toLowerCase().includes('canva')) {
        appDisplay = 'Canva';
      }
      // Kürze Adobe Firefly-Anzeige
      else if (appDisplay.toLowerCase().includes('adobe firefly') || appDisplay.toLowerCase().includes('adobe_firefly')) {
        appDisplay = 'Adobe Firefly';
      }
    } else if (s.software) {
      appDisplay = s.software;
    }
    if (appDisplay !== '—') {
      html += '<div class="wp-c2pa-combined-field">' + t('usedApplication') + '</div>';
      html += '<div class="wp-c2pa-combined-value">' + appDisplay + '</div>';
    }
    
    // Abmessungen (aus allen verfügbaren Quellen)
    let dimensions = '—';
    if (s.dimensions) {
      dimensions = s.dimensions;
    } else if (meta && meta.raw) {
      if (meta.raw.ImageWidth && meta.raw.ImageHeight) {
        dimensions = meta.raw.ImageWidth + '×' + meta.raw.ImageHeight;
      }
    } else if (el) {
      const w = el.naturalWidth || el.width;
      const h = el.naturalHeight || el.height;
      if (w && h) {
        dimensions = w + '×' + h;
      }
    }
    if (dimensions !== '—') {
      html += '<div class="wp-c2pa-combined-field">' + t('dimensions') + '</div>';
      html += '<div class="wp-c2pa-combined-value">' + dimensions + '</div>';
    }
    
    // Bearbeitung (nur wenn C2PA-Daten vorhanden)
    if (man && man.actions && Array.isArray(man.actions) && man.actions.length > 0) {
      html += '<div class="wp-c2pa-combined-field">' + t('editing') + '</div>';
      html += '<div class="wp-c2pa-combined-value wp-c2pa-edit-count">' + man.actions.length + ' ' + (cfg.settings.language === 'en' ? 'Changes' : 'Änderungen') + '</div>';
    }
    
    // Ebenen (aus allen verfügbaren Quellen)
    let layerCount = null;
    if (man && man.ingredients && Array.isArray(man.ingredients)) {
      layerCount = man.ingredients.length;
    } else if (man && man.actionsMeta && man.actionsMeta.layerCount) {
      layerCount = man.actionsMeta.layerCount;
    } else if (man && man.actionsMeta && man.actionsMeta.layers && Array.isArray(man.actionsMeta.layers)) {
      layerCount = man.actionsMeta.layers.length;
    }
    if (layerCount) {
      html += '<div class="wp-c2pa-combined-field">' + t('layers') + '</div>';
      const layerText = (layerCount === 1) ? (cfg.settings.language === 'en' ? '1 Layer' : '1 Ebene') : layerCount + ' ' + (cfg.settings.language === 'en' ? 'Layers' : 'Ebenen');
      html += '<div class="wp-c2pa-combined-value wp-c2pa-edit-count">' + layerText + '</div>';
    }
    
    // Beschreibung (aus allen verfügbaren Quellen)
    let description = '—';
    if (meta && meta.raw && meta.raw.Description) {
      description = meta.raw.Description;
    } else if (s.description) {
      description = s.description;
    }
    if (description !== '—') {
      html += '<div class="wp-c2pa-combined-field">' + t('description') + '</div>';
      html += '<div class="wp-c2pa-combined-value">' + description + '</div>';
    }
    
    // Erstellungszeit (aus allen verfügbaren Quellen)
    let creationTime = '—';
    if (man && man.signatureInfo && man.signatureInfo.time) {
      creationTime = new Date(man.signatureInfo.time).toLocaleString('de-DE');
    } else if (s.createdAt) {
      creationTime = formatDate(s.createdAt);
    } else if (meta && meta.raw && meta.raw['Creation Time']) {
      creationTime = formatDate(meta.raw['Creation Time']);
    }
    if (creationTime !== '—') {
      html += '<div class="wp-c2pa-combined-field">' + t('creationTime') + '</div>';
      html += '<div class="wp-c2pa-combined-value">' + creationTime + '</div>';
    }
    
    // Kamera (aus allen verfügbaren Quellen)
    let camera = '—';
    if (s.cameraMake || s.cameraModel) {
      camera = (s.cameraMake || '') + ' ' + (s.cameraModel || '').trim();
    } else if (meta && meta.raw && meta.raw.Make && meta.raw.Model) {
      camera = meta.raw.Make + ' ' + meta.raw.Model;
    }
    if (camera !== '—' && camera.trim() !== '') {
      html += '<div class="wp-c2pa-combined-field">' + t('camera') + '</div>';
      html += '<div class="wp-c2pa-combined-value">' + camera.trim() + '</div>';
    }
    
    // Objektiv (aus allen verfügbaren Quellen)
    let lens = '—';
    if (s.lensModel) {
      lens = s.lensModel;
    } else if (meta && meta.raw && meta.raw.LensModel) {
      lens = meta.raw.LensModel;
    }
    if (lens !== '—') {
      html += '<div class="wp-c2pa-combined-field">' + t('lens') + '</div>';
      html += '<div class="wp-c2pa-combined-value">' + lens + '</div>';
    }
    
    html += '</div>';
    html += '</div>';
    
    if(!html) html='<div class="wp-c2pa-combined-box-sub">' + t('noSummaryAvailable') + '</div>';
    return html;
  }

  function renderExtended(man, meta){ 
    const s=(meta&&meta.summary)||{}; 
    let html='';
    
    // KI-Indikator prüfen (an erster Stelle)
    try{ 
      const rawDataString = safeJSON({manifest:man, metadata:meta},0) || '';
                  const hasIptc = rawDataString.toLowerCase().includes('iptc.org');
            const hasMidjourneyVersion = rawDataString.match(/--v\s+[\d.]+/);
            const hasMidjourney = rawDataString.toLowerCase().includes('midjourney');
                  const hasOpenAI = rawDataString.toLowerCase().includes('openai');
            const hasChatGPT = rawDataString.toLowerCase().includes('chatgpt');
            const hasSora = rawDataString.toLowerCase().includes('sora');
            const hasAdobeFirefly = rawDataString.toLowerCase().includes('adobe_firefly');
            const hasCanva = rawDataString.toLowerCase().includes('canva');
            const hasAdobePhotoshop = (rawDataString.toLowerCase().includes('adobe photoshop') || rawDataString.toLowerCase().includes('adobe_photoshop')) && !rawDataString.toLowerCase().includes('lightroom');
            const hasAdobeLightroom = rawDataString.toLowerCase().includes('lightroom');
            const hasGeminiFlash = rawDataString.toLowerCase().includes('gemini-flash');
            const hasNanoBanana = rawDataString.toLowerCase().includes('nano-banana');
      
              if(hasIptc || hasMidjourneyVersion || hasMidjourney || hasOpenAI || hasChatGPT || hasSora || hasAdobeFirefly || hasCanva || hasAdobePhotoshop || hasAdobeLightroom || hasGeminiFlash || hasNanoBanana){
        html+=`<div class="wp-c2pa-ai-indicator">`;
        if(hasIptc){
          html+=`<div class="wp-c2pa-ai-badge">${t('aiImage')}</div>`;
        }
                      if(hasMidjourneyVersion){
                html+=`<div class="wp-c2pa-midjourney-badge">Midjourney</div>`;
              }
              if(hasMidjourney){
                html+=`<div class="wp-c2pa-midjourney-badge">Midjourney</div>`;
              }
        if(hasOpenAI){
          html+=`<div class="wp-c2pa-openai-badge">OpenAI</div>`;
        }
        if(hasChatGPT){
          html+=`<div class="wp-c2pa-chatgpt-badge">ChatGPT</div>`;
        }
                      if(hasSora){
                html+=`<div class="wp-c2pa-sora-badge">Sora</div>`;
              }
              if(hasAdobeFirefly){
                html+=`<div class="wp-c2pa-adobe-firefly-badge">Adobe Firefly</div>`;
              }
              if(hasCanva){
                html+=`<div class="wp-c2pa-canva-badge">Canva</div>`;
              }
              if(hasAdobePhotoshop){
                html+=`<div class="wp-c2pa-adobe-photoshop-badge">Adobe Photoshop</div>`;
              }
              if(hasAdobeLightroom){
                html+=`<div class="wp-c2pa-adobe-lightroom-badge">Adobe Lightroom</div>`;
              }
              if(hasGeminiFlash){
                html+=`<div class="wp-c2pa-gemini-flash-badge">gemini-flash</div>`;
              }
              if(hasNanoBanana){
                html+=`<div class="wp-c2pa-nano-banana-badge">nano-banana</div>`;
              }
        html+=`</div>`;
      }
    }catch{}
    
    // Basis-Informationen
    if(man){
      html += '<div class="wp-c2pa-combined-section">';
      html += '<div class="wp-c2pa-combined-section-title">' + (cfg.settings.language === 'en' ? 'Basic Information' : 'Basis-Informationen') + '</div>';
      html += '<div class="wp-c2pa-combined-grid">';
      
      // Name der Datei
      if(man.title) {
        html += '<div class="wp-c2pa-combined-field">' + t('fileName') + '</div>';
        html += '<div class="wp-c2pa-combined-value">' + man.title + '</div>';
      }
      
      // Ausgestellt von
      if(man.creator && man.creator.name) {
        html += '<div class="wp-c2pa-combined-field">' + t('issuedBy') + '</div>';
        html += '<div class="wp-c2pa-combined-value">' + man.creator.name + '</div>';
      }
      
      // Ausgestellt am
      if(man.signatureInfo && man.signatureInfo.time) {
        html += '<div class="wp-c2pa-combined-field">' + t('creationTime') + '</div>';
        html += '<div class="wp-c2pa-combined-value">' + new Date(man.signatureInfo.time).toLocaleString(cfg.settings.language === 'en' ? 'en-US' : 'de-DE') + '</div>';
      }
      
      // Verwendete Applikation
      if(man.claimGenerator) {
        html += '<div class="wp-c2pa-combined-field">' + t('usedApplication') + '</div>';
        
        // Kürze Photoshop-Anzeige
        let appDisplay = man.claimGenerator;
        if (appDisplay.toLowerCase().includes('adobe_photoshop') || appDisplay.toLowerCase().includes('adobe photoshop')) {
          // Suche nach Version (z.B. "26.8.1")
          const versionMatch = appDisplay.match(/(\d+\.\d+\.\d+)/);
          if (versionMatch) {
            appDisplay = 'Adobe Photoshop ' + versionMatch[1];
          } else {
            // Fallback: Entferne nur die C2PA-Bibliotheken
            appDisplay = appDisplay.replace(/\s+adobe_c2pa\/[^\s]+/g, '').replace(/\s+c2pa-rs\/[^\s]+/g, '');
            appDisplay = appDisplay.replace('Adobe_Photoshop', 'Adobe Photoshop');
          }
        }
        // Kürze Canva-Anzeige
        else if (appDisplay.toLowerCase().includes('canva')) {
          appDisplay = 'Canva';
        }
        // Kürze Adobe Firefly-Anzeige
        else if (appDisplay.toLowerCase().includes('adobe firefly') || appDisplay.toLowerCase().includes('adobe_firefly')) {
          appDisplay = 'Adobe Firefly';
        }
        
        html += '<div class="wp-c2pa-combined-value">' + appDisplay + '</div>';
      }
      
      // Abmessungen
      if(s.dimensions) {
        html += '<div class="wp-c2pa-combined-field">' + t('dimensions') + '</div>';
        html += '<div class="wp-c2pa-combined-value">' + s.dimensions + '</div>';
      }
      
      // Anzahl der Bearbeitungsschritte
      if(man.actions && Array.isArray(man.actions) && man.actions.length > 0) {
        html += '<div class="wp-c2pa-combined-field">' + t('editing') + '</div>';
        html += '<div class="wp-c2pa-combined-value wp-c2pa-edit-count">' + man.actions.length + ' ' + (cfg.settings.language === 'en' ? 'Changes' : 'Änderungen') + '</div>';
      }
      
      // Anzahl der Ebenen - EINFACHE UND DIREKTE ERKENNUNG
      let layerCount = null;
      
      // 1. Direkte Ebenen-Informationen
      if (man.actionsMeta && man.actionsMeta.layerCount) {
        layerCount = man.actionsMeta.layerCount;
      } else if (man.actionsMeta && man.actionsMeta.layers && Array.isArray(man.actionsMeta.layers)) {
        layerCount = man.actionsMeta.layers.length;
      }
      
      // 2. Suche in Aktionen nach Ebenen
      if (!layerCount && man.actions && Array.isArray(man.actions)) {
        for (let action of man.actions) {
          if (action && action.parameters) {
            // Suche nach verschiedenen Ebenen-Parametern
            if (action.parameters.layerCount) {
              layerCount = action.parameters.layerCount;
              break;
            } else if (action.parameters.layers) {
              if (Array.isArray(action.parameters.layers)) {
                layerCount = action.parameters.layers.length;
                break;
              } else if (typeof action.parameters.layers === 'number') {
                layerCount = action.parameters.layers;
                break;
              }
            } else if (action.parameters['com.adobe.layer']) {
              layerCount = action.parameters['com.adobe.layer'];
              break;
            }
          }
        }
      }
      
      // 3. Suche in Rohdaten nach "X Ebenen" oder "X layers"
      if (!layerCount) {
        try {
          const rawDataString = safeJSON({manifest:man, metadata:meta},0) || '';
          const layerMatch = rawDataString.match(/(\d+)\s*(?:ebenen?|layers?)/i);
          if (layerMatch) {
            layerCount = parseInt(layerMatch[1]);
          }
        } catch (e) {
          // Ignoriere Fehler
        }
      }
      
      // 4. Zähle alle Aktionen, die mit Ebenen zu tun haben (Fallback)
      if (!layerCount && man.actions && Array.isArray(man.actions)) {
        const layerActions = man.actions.filter(action => 
          action.action && (
            action.action.toLowerCase().includes('layer') || 
            action.action.toLowerCase().includes('ebene') ||
            action.action.toLowerCase().includes('duplicate') ||
            action.action.toLowerCase().includes('new layer')
          )
        );
        if (layerActions.length > 0) {
          layerCount = layerActions.length;
        }
      }
      
      // Zeige Ebenen an, wenn gefunden
      if (layerCount) {
        html += '<div class="wp-c2pa-combined-field">' + t('layers') + '</div>';
        const layerText = (layerCount === 1) ? (cfg.settings.language === 'en' ? '1 Layer' : '1 Ebene') : layerCount + ' ' + (cfg.settings.language === 'en' ? 'Layers' : 'Ebenen');
        html += '<div class="wp-c2pa-combined-value">' + layerText + '</div>';
      }
      
      // BESTANDTEILE - DIREKTE ERKENNUNG AUS DEN INGREDIENTS (EBENEN)
      let componentCount = null;
      
      // 1. Direkte Zählung der Ingredients (Ebenen) - das funktioniert bereits!
      if (man.ingredients && Array.isArray(man.ingredients)) {
        componentCount = man.ingredients.length;
      }
      
      // 2. Fallback: Suche in actionsMeta
      if (!componentCount && man.actionsMeta) {
        if (man.actionsMeta.componentCount) {
          componentCount = man.actionsMeta.componentCount;
        } else if (man.actionsMeta.components && Array.isArray(man.actionsMeta.components)) {
          componentCount = man.actionsMeta.components.length;
        } else if (man.actionsMeta.layerCount) {
          componentCount = man.actionsMeta.layerCount;
        } else if (man.actionsMeta.layers && Array.isArray(man.actionsMeta.layers)) {
          componentCount = man.actionsMeta.layers.length;
        }
      }
      
      // 3. Fallback: Suche in actions
      if (!componentCount && man.actions && Array.isArray(man.actions)) {
        for (let action of man.actions) {
          if (action && action.parameters) {
            if (action.parameters.componentCount) {
              componentCount = action.parameters.componentCount;
              break;
            } else if (action.parameters.components) {
              if (Array.isArray(action.parameters.components)) {
                componentCount = action.parameters.components.length;
                break;
              } else if (typeof action.parameters.components === 'number') {
                componentCount = action.parameters.components;
                break;
              }
            } else if (action.parameters.layerCount) {
              componentCount = action.parameters.layerCount;
              break;
            } else if (action.parameters.layers) {
              if (Array.isArray(action.parameters.layers)) {
                componentCount = action.parameters.layers.length;
                break;
              } else if (typeof action.parameters.layers === 'number') {
                componentCount = action.parameters.layers;
                break;
              }
            }
          }
        }
      }
      
      // Zeige Ebenen an, wenn gefunden
      if (componentCount) {
        html += '<div class="wp-c2pa-combined-field">' + t('layers') + '</div>';
        const layerText = (componentCount === 1) ? (cfg.settings.language === 'en' ? '1 Layer' : '1 Ebene') : componentCount + ' ' + (cfg.settings.language === 'en' ? 'Layers' : 'Ebenen');
        html += '<div class="wp-c2pa-combined-value wp-c2pa-edit-count">' + layerText + '</div>';
      }
      
      html += '</div>';
      html += '</div>';
    }
    
    // Nur normale Metadaten (EXIF/IPTC/XMP) - jetzt wieder aufklappbar
    if(Object.keys(s).length){ 
      html+=`<div class="wp-c2pa-combined-section"><div class="wp-c2pa-combined-section-title">${cfg.settings.language === 'en' ? 'Metadata' : 'Metadaten'}</div>
        <div class="wp-c2pa-combined-box">
          <div class="wp-c2pa-combined-box-title">${cfg.settings.language === 'en' ? 'EXIF/IPTC/XMP Data' : 'EXIF/IPTC/XMP Daten'}</div>
          <details><summary>${cfg.settings.language === 'en' ? 'Show Breakdown' : 'Aufschlüsselung anzeigen'}</summary>
            <div class="wp-c2pa-combined-grid">
                <div><div class="wp-c2pa-combined-field">${cfg.settings.language === 'en' ? 'Created' : 'Erstellt am'}</div><div class="wp-c2pa-combined-value">${s.createdAt?formatDate(s.createdAt):'—'}</div></div>
                <div><div class="wp-c2pa-combined-field">${cfg.settings.language === 'en' ? 'Modified' : 'Bearbeitet am'}</div><div class="wp-c2pa-combined-value">${s.modifiedAt?formatDate(s.modifiedAt):'—'}</div></div>
                <div><div class="wp-c2pa-combined-field">${cfg.settings.language === 'en' ? 'Software' : 'Software'}</div><div class="wp-c2pa-combined-value">${s.software||'—'}</div></div>
                <div><div class="wp-c2pa-combined-field">${cfg.settings.language === 'en' ? 'Artist' : 'Urheber'}</div><div class="wp-c2pa-combined-value">${s.artist||'—'}</div></div>
                <div><div class="wp-c2pa-combined-field">${cfg.settings.language === 'en' ? 'Camera' : 'Kamera'}</div><div class="wp-c2pa-combined-value">${(s.cameraMake||'')} ${(s.cameraModel||'')}</div></div>
                <div><div class="wp-c2pa-combined-field">${cfg.settings.language === 'en' ? 'Dimensions' : 'Abmessungen'}</div><div class="wp-c2pa-combined-value">${s.dimensions||'—'}</div></div>
              </div>
          </details>
        </div>
      </div>`; 
    }

    // Zusätzliche Rohdaten-Felder (falls vorhanden) - jetzt wieder aufklappbar
    const raw = (meta && meta.raw) || {};
    const rawKeys = ['ImageWidth','ImageHeight','BitDepth','ColorType','Compression','Filter','Interlace','Creation Time','Author','Description'];
    const present = rawKeys.map(k=>[k, raw[k]]).filter(([,v])=> v!==undefined && v!==null && String(v)!=='');
    if(present.length){
      const rows = present.map(([k,v])=>{
        // Schöne Labels
        const labelMap = cfg.settings.language === 'en' ? 
          { ImageWidth:'Width (px)', ImageHeight:'Height (px)', BitDepth:'Color Depth (Bit)', ColorType:'Color Type', Compression:'Compression', Filter:'Filter', Interlace:'Interlace', 'Creation Time':'Creation Time', Author:'Author', Description:'Description' } :
          { ImageWidth:'Breite (px)', ImageHeight:'Höhe (px)', BitDepth:'Farbtiefe (Bit)', ColorType:'Farbtyp', Compression:'Kompression', Filter:'Filter', Interlace:'Interlace', 'Creation Time':'Erstellungszeit', Author:'Autor', Description:'Beschreibung' };
        const label = labelMap[k]||k;
        // Werte formatieren
        const val = (k==='Creation Time') ? formatDate(v) : String(v);
        return `<div><div class=\"wp-c2pa-combined-field\">${label}</div><div class=\"wp-c2pa-combined-value\">${val}</div></div>`;
      }).join('');
      html+=`<div class="wp-c2pa-combined-section"><div class="wp-c2pa-combined-section-title">${cfg.settings.language === 'en' ? 'Raw Data – Technical Fields' : 'Rohdaten – Technische Felder'}</div>
        <div class="wp-c2pa-combined-box">
          <div class="wp-c2pa-combined-box-title">${cfg.settings.language === 'en' ? 'Technical Image Data' : 'Technische Bilddaten'}</div>
          <details><summary>${cfg.settings.language === 'en' ? 'Show Breakdown' : 'Aufschlüsselung anzeigen'}</summary>
            <div class="wp-c2pa-combined-grid">${rows}</div>
          </details>
        </div>
      </div>`;
    }

    // Photoshop-Bearbeitungsschritte wurden entfernt und in Basis-Informationen verschoben
    
    if(!html) html='<div class="wp-c2pa-combined-box-sub">' + (cfg.settings.language === 'en' ? 'No extended data.' : 'Keine erweiterten Daten.') + '</div>';
    return html;
  }

  function renderAdvancedView(man, meta) {
    if (!man) return '<div class="wp-c2pa-combined-box-sub">Keine erweiterten Daten verfügbar.</div>';
    
    let html = '';
    
    // Ebenen (Ingredients) - nur anzeigen wenn vorhanden
    if (man.ingredients && Array.isArray(man.ingredients) && man.ingredients.length > 0) {
      html += '<div class="wp-c2pa-combined-section">';
      html += '<div class="wp-c2pa-combined-section-title">' + t('layers') + '</div>';
      html += '<div class="wp-c2pa-combined-box">';
      const layerText = cfg.settings.language === 'en' ? 
        (man.ingredients.length === 1 ? '1 Layer' : man.ingredients.length + ' Layers') :
        (man.ingredients.length === 1 ? '1 Ebene' : man.ingredients.length + ' Ebenen');
      html += '<div class="wp-c2pa-combined-box-title">' + layerText + '</div>';
      html += '<details><summary>' + t('showDetails') + '</summary>';
      html += '<div class="wp-c2pa-combined-stack">';
      
      man.ingredients.forEach((ingredient, index) => {
        html += '<div class="wp-c2pa-combined-box" style="margin-bottom: 8px;">';
        const layerTitle = cfg.settings.language === 'en' ? 'Layer ' + (index + 1) : 'Ebene ' + (index + 1);
        html += '<div class="wp-c2pa-combined-box-title">' + layerTitle + '</div>';
        html += '<div class="wp-c2pa-combined-grid">';
        
        if (ingredient.title) {
          html += '<div><div class="wp-c2pa-combined-field">' + (cfg.settings.language === 'en' ? 'Title' : 'Titel') + '</div><div class="wp-c2pa-combined-value">' + ingredient.title + '</div></div>';
        }
        if (ingredient.format) {
          html += '<div><div class="wp-c2pa-combined-field">' + (cfg.settings.language === 'en' ? 'Format' : 'Format') + '</div><div class="wp-c2pa-combined-value">' + ingredient.format + '</div></div>';
        }
        if (ingredient.claimGenerator) {
          html += '<div><div class="wp-c2pa-combined-field">' + (cfg.settings.language === 'en' ? 'Created with' : 'Erstellt mit') + '</div><div class="wp-c2pa-combined-value">' + ingredient.claimGenerator + '</div></div>';
        }
        if (ingredient.hash) {
          html += '<div><div class="wp-c2pa-combined-field">' + (cfg.settings.language === 'en' ? 'Hash' : 'Hash') + '</div><div class="wp-c2pa-combined-value">' + ingredient.hash + '</div></div>';
        }
        
        html += '</div>';
        html += '</div>';
      });
      
      html += '</div>';
      html += '</details>';
      html += '</div>';
      html += '</div>';
    }
    
    // Bearbeitungsschritte (Actions) - nur anzeigen wenn vorhanden
    if (man.actions && Array.isArray(man.actions) && man.actions.length > 0) {
      html += '<div class="wp-c2pa-combined-section">';
      html += '<div class="wp-c2pa-combined-section-title">' + t('editingSteps') + '</div>';
      html += '<div class="wp-c2pa-combined-box">';
      const stepText = cfg.settings.language === 'en' ? 
        (man.actions.length === 1 ? '1 Editing Step' : man.actions.length + ' Editing Steps') :
        (man.actions.length === 1 ? '1 Bearbeitungsschritt' : man.actions.length + ' Bearbeitungsschritte');
      html += '<div class="wp-c2pa-combined-box-title">' + stepText + '</div>';
      html += '<details><summary>' + t('showDetails') + '</summary>';
      html += '<div class="wp-c2pa-combined-stack">';
      
      man.actions.forEach((action, index) => {
        html += '<div class="wp-c2pa-combined-box" style="margin-bottom: 8px;">';
        const stepTitle = cfg.settings.language === 'en' ? 'Step ' + (index + 1) : 'Schritt ' + (index + 1);
        html += '<div class="wp-c2pa-combined-box-title">' + stepTitle + '</div>';
        html += '<div class="wp-c2pa-combined-grid">';
        
        // Zeige nur die description an, falls vorhanden
        if (action.parameters && action.parameters.description) {
          html += '<div><div class="wp-c2pa-combined-field">' + t('description') + '</div><div class="wp-c2pa-combined-value">' + action.parameters.description + '</div></div>';
        } else if (action.action) {
          // Fallback: Zeige action an, falls keine description vorhanden
          html += '<div><div class="wp-c2pa-combined-field">' + (cfg.settings.language === 'en' ? 'Action' : 'Aktion') + '</div><div class="wp-c2pa-combined-value">' + action.action + '</div></div>';
        }
        
        html += '</div>';
        html += '</div>';
      });
      
      html += '</div>';
      html += '</details>';
      html += '</div>';
      html += '</div>';
    }
    
    if (!html) {
      html = '<div class="wp-c2pa-combined-box-sub">' + (cfg.settings.language === 'en' ? 'No extended data available.' : 'Keine erweiterten Daten verfügbar.') + '</div>';
    }
    
    return html;
  }

  function ensureModal(){ 
    if(document.getElementById('wp-c2pa-combined-modal')) return; 
    const m=document.createElement('div'); 
    m.id='wp-c2pa-combined-modal'; 
    // Tabs basierend auf Einstellungen erstellen
         let tabButtons = `<button class="wp-c2pa-combined-tab is-active" data-tab="summary">${t('summary')}</button>`;
    
    // Rohdaten-Tab nur hinzufügen wenn aktiviert
    if (cfg.settings.showRawData) {
        tabButtons += `<button class="wp-c2pa-combined-tab" data-tab="raw">${cfg.settings.strings?.rawData||'Rohdaten'}</button>`;
    }
    
    let tabPanels = `<div class="wp-c2pa-combined-tabpanel" data-panel="summary"></div>`;
    
    // Rohdaten-Panel nur hinzufügen wenn aktiviert
    if (cfg.settings.showRawData) {
        tabPanels += `<div class="wp-c2pa-combined-tabpanel is-hidden" data-panel="raw"><pre class="wp-c2pa-combined-raw"></pre></div>`;
    }
    
    m.innerHTML=`<div class="wp-c2pa-combined-modal-backdrop" data-close></div><div class="wp-c2pa-combined-modal"><div class="wp-c2pa-combined-modal-header"><div class="wp-c2pa-combined-modal-title">${cfg.settings.strings?.contentCredentials||'Content Credentials'}</div><button class="wp-c2pa-combined-modal-close" type="button" aria-label="Schließen" data-close>&times;</button></div><div class="wp-c2pa-combined-tabs">${tabButtons}</div><div class="wp-c2pa-combined-modal-body">${tabPanels}</div></div>`; 
    document.body.appendChild(m); 
    m.addEventListener('click',e=>{ if(e.target&&e.target.hasAttribute('data-close')) hideModal(); }); 
    m.querySelectorAll('.wp-c2pa-combined-tab').forEach(btn=>btn.addEventListener('click',()=>{ 
      const tab=btn.getAttribute('data-tab'); 
      m.querySelectorAll('.wp-c2pa-combined-tab').forEach(b=>b.classList.toggle('is-active',b===btn)); 
      m.querySelectorAll('.wp-c2pa-combined-tabpanel').forEach(p=>p.classList.toggle('is-hidden',p.getAttribute('data-panel')!==tab)); 
    })); 
    document.addEventListener('keydown',e=>{ if(e.key==='Escape') hideModal(); }); 
  }
  
  function showModal(){ 
    // Speichere aktuelle Scroll-Position für IE/Edge-Kompatibilität
    const scrollY = window.pageYOffset || document.documentElement.scrollTop || document.body.scrollTop || 0;
    
    // Setze body auf fixed Position um Scroll zu verhindern
    document.body.style.position = 'fixed';
    document.body.style.top = `-${scrollY}px`;
    document.body.style.width = '100%';
    
    document.body.classList.add('wp-c2pa-combined-modal-open'); 
    const el=document.getElementById('wp-c2pa-combined-modal'); 
    if(el) el.classList.add('is-open'); 
    
    // Speichere Scroll-Position für Wiederherstellung
    el.dataset.scrollY = scrollY;
    
    // Erzwinge Scroll-Funktionalität im Modal-Body
    setTimeout(() => {
      const modalBody = el.querySelector('.wp-c2pa-combined-modal-body');
      if (modalBody) {
        // Stelle sicher, dass das Element scrollbar ist
        modalBody.style.overflowY = 'auto';
        modalBody.style.height = 'calc(85vh - 120px)';
        
        // Füge Wheel-Event-Listener hinzu für bessere Trackpad-Unterstützung
        modalBody.addEventListener('wheel', function(e) {
          e.stopPropagation();
        }, { passive: true });
      }
    }, 100);
  }
  
  function hideModal(){ 
    document.body.classList.remove('wp-c2pa-combined-modal-open'); 
    const el=document.getElementById('wp-c2pa-combined-modal'); 
    
    // Stelle Scroll-Position wieder her (IE/Edge-kompatibel)
    if(el && el.dataset.scrollY) {
      const scrollY = parseInt(el.dataset.scrollY);
      
      // Entferne fixed Position
      document.body.style.position = '';
      document.body.style.top = '';
      document.body.style.width = '';
      
      // Stelle Scroll-Position wieder her
      window.scrollTo(0, scrollY);
      
      // Lösche gespeicherte Position
      delete el.dataset.scrollY;
    } else {
      // Fallback für den Fall, dass keine Position gespeichert wurde
      document.body.style.position = '';
      document.body.style.top = '';
      document.body.style.width = '';
    }
    
    if(el) el.classList.remove('is-open'); 
  }
  
  // Toggle function wurde entfernt, da Photoshop-Bearbeitungsschritte nicht mehr angezeigt werden

  function attachBadge(el, man, meta){ 
    debug('info', 'attachBadge gestartet', {
      src: el.src || el.dataset.src,
      className: el.className,
      hasManifest: !!man,
      hasMetadata: !!meta
    });
    
    // NEUE LÖSUNG: Globale Badge-Verwaltung
    const badgeId = generateBadgeId(el);
    
    // Prüfe ob dieser Badge bereits global existiert
    if (cfg.attachedBadges.has(badgeId)) {
      debug('info', 'Badge bereits global registriert, überspringe', badgeId);
      return;
    }
    
    // Prüfe ob bereits ein Badge existiert (verhindert doppelte Badges)
    if (el.nextElementSibling && el.nextElementSibling.classList.contains('wp-c2pa-combined-badge')) {
      debug('info', 'Badge bereits vorhanden, überspringe');
      return;
    }
    
    // Prüfe ob bereits ein Badge-Container existiert
    if (el.parentElement && el.parentElement.classList.contains('wp-c2pa-combined-badge-container')) {
      const existingBadge = el.parentElement.querySelector('.wp-c2pa-combined-badge');
      if (existingBadge) {
        debug('info', 'Badge bereits in Container vorhanden, überspringe');
        return;
    }
    }
    
    // Prüfe ob das Element ein UI-Element ist (Icon, Logo, etc.)
    if (isUIElement(el)) {
      debug('info', 'UI-Element erkannt, überspringe');
      return; // Kein Badge für UI-Elemente
    }
    
    // Prüfe ob Titelbilder ausgeschlossen werden sollen
    if (cfg.settings.excludeFeaturedImages && isFeaturedImage(el)) {
      debug('info', 'Titelbild erkannt und Badge ausgeschlossen', {
        src: el.src,
        className: el.className
      });
      return; // Kein Badge für Titelbilder
    }
    
    // Prüfe ob die Startseite ausgeschlossen werden soll
    if (cfg.settings.excludeHomepage && isHomepage()) {
      debug('info', 'Startseite erkannt und Badge ausgeschlossen', {
        src: el.src,
        className: el.className,
        currentPath: window.location.pathname
      });
      return; // Kein Badge auf der Startseite
    }
    
    // Wähle zwischen den beiden Badge-Positionierungsmethoden
    const positioningMethod = cfg.settings.themeCompatibilityMode || 'standard';
    
    if (positioningMethod === 'minimal') {
      // MINIMALE METHODE: Exakte Logik von Version 003205
      attachBadgeMinimal(el, man, meta);
    } else {
      // STANDARD METHODE: Container wird erstellt, Badge schwebt über dem Bild
      attachBadgeStandard(el, man, meta);
    }
  }
  
  // MINIMALE METHODE: Exakte Logik von Version 003205
  function attachBadgeMinimal(el, man, meta) {
    // Badge-Existenz wird bereits in attachBadge() geprüft
    
    // Setze das Bild auf relative Position für Badge-Positionierung
    if (el.style.position !== 'absolute' && el.style.position !== 'fixed') {
      el.style.position = 'relative';
    }
    
    const b = document.createElement('button'); 
    b.type = 'button'; 
    b.className = 'wp-c2pa-combined-badge'; 
    
    // Positionierung basierend auf Einstellung
    const position = cfg.settings.badgePosition || 'bottom-left';
    b.style.position = 'absolute';
    b.style.zIndex = '10';
    
    // Proportionale Badge-Positionierung basierend auf Bildgröße
    const imageRect = el.getBoundingClientRect();
    const imageWidth = imageRect.width;
    const imageHeight = imageRect.height;
    
    // Berechne proportionale Abstände (5% der Bildgröße, mindestens 8px, maximal 20px)
    const marginX = Math.max(8, Math.min(20, Math.round(imageWidth * 0.05)));
    const marginY = Math.max(8, Math.min(20, Math.round(imageHeight * 0.05)));
    
    switch(position) {
      case 'bottom-left':
        b.style.left = marginX + 'px';
        b.style.bottom = marginY + 'px';
        break;
      case 'bottom-right':
        b.style.right = marginX + 'px';
        b.style.bottom = marginY + 'px';
        break;
      case 'top-left':
        b.style.left = marginX + 'px';
        b.style.top = marginY + 'px';
        break;
      case 'top-right':
        b.style.right = marginX + 'px';
        b.style.top = marginY + 'px';
        break;
    }
    
    // Apply custom badge styling
    const iconMap = {
      'shield': '🛡️',
      'check': '✅',
      'lock': '🔒',
      'star': '⭐',
      'verified': '✓',
      'certificate': '📜',
      'eye': '👁️',
      'info': 'ℹ️',
      'badge': '🏷️',
      'security': '🔐',
      'eye-shield': '<img src="' + pluginUrl + 'assets/images/eye_shield_badge_transparent.svg" style="width: 16px; height: 16px; vertical-align: middle;" alt="Eye Shield">'
    };
    
    const badgeColor = cfg.settings.badgeColor || '#0073aa';
    const badgeIcon = cfg.settings.badgeIcon || 'shield';
    const badgeText = cfg.settings.badgeText || 'Content Credentials';
    const badgeSize = cfg.settings.badgeSize || 'medium';
    const badgeStyle = cfg.settings.badgeStyle || 'solid';
    const badgeDisplay = cfg.settings.badgeDisplay || 'icon_text';
    // const badgeOpacity = cfg.settings.badgeOpacity || '1.0'; // DEAKTIVIERT - Badge ist immer 90% transparent
    const badgeShadow = cfg.settings.badgeShadow || 'light';
    const badgeBorderRadius = cfg.settings.badgeBorderRadius || '8';
    const badgePadding = cfg.settings.badgePadding || '8';
    const badgeFontSize = cfg.settings.badgeFontSize || '14';
    const badgeFontWeight = cfg.settings.badgeFontWeight || '600';
    
    // Add size and style classes - CSS hat Vorrang vor Inline-Styles
    b.classList.add('badge-size-' + badgeSize);
    b.classList.add('badge-style-' + badgeStyle);
    
    // KEINE Transparenz-Logik mehr - Badge ist immer 90% transparent
    // b.classList.add('transparent'); // DEAKTIVIERT
    debug('info', 'Badge wird immer mit 90% Transparenz angezeigt');
    
    // KEINE Inline-Styles mehr - CSS-Klassen übernehmen die Formatierung
    // Nur noch die notwendigen Positionierungs-Styles
    
    // Apply new styling options (nur wenn nicht durch CSS überschrieben)
    // b.style.opacity = badgeOpacity; // DEAKTIVIERT - CSS übernimmt Transparenz
    b.style.borderRadius = badgeBorderRadius + 'px';
    b.style.fontWeight = badgeFontWeight;
    
    // Apply shadow effects
    switch(badgeShadow) {
      case 'none':
        b.style.boxShadow = 'none';
        break;
      case 'light':
        b.style.boxShadow = '0 2px 4px rgba(0,0,0,0.1)';
        break;
      case 'medium':
        b.style.boxShadow = '0 4px 8px rgba(0,0,0,0.15)';
        break;
      case 'strong':
        b.style.boxShadow = '0 8px 16px rgba(0,0,0,0.2)';
        break;
    }
    
    // Set display content based on setting
    if (badgeDisplay === 'icon_only') {
      if (badgeIcon === 'eye-shield') {
        b.innerHTML = iconMap[badgeIcon];
      } else {
        b.textContent = iconMap[badgeIcon] || '🛡️';
      }
    } else if (badgeDisplay === 'text_only') {
      b.textContent = badgeText;
    } else { // icon_text
      if (badgeIcon === 'eye-shield') {
        b.innerHTML = iconMap[badgeIcon] + ' ' + badgeText;
      } else {
        b.textContent = (iconMap[badgeIcon] || '🛡️') + ' ' + badgeText;
      }
    }
    
    // RADIKALE LÖSUNG: Entferne ALLE Inline-Styles die CSS überschreiben könnten
    setTimeout(() => {
        // CSS-Klassen haben Vorrang vor Inline-Styles
        if (b.style.fontSize) b.style.removeProperty('font-size');
        if (b.style.padding) b.style.removeProperty('padding');
        if (b.style.background) b.style.removeProperty('background');
        if (b.style.backgroundColor) b.style.removeProperty('background-color');
        if (b.style.border) b.style.removeProperty('border');
        if (b.style.color) b.style.removeProperty('color');
        
        // Zusätzliche Sicherheit: Entferne alle möglichen CSS-Eigenschaften
        const propertiesToRemove = ['background', 'background-color', 'border', 'color', 'font-size', 'padding'];
        propertiesToRemove.forEach(prop => {
            if (b.style[prop]) {
                b.style.removeProperty(prop);
                debug('info', 'Inline-Style entfernt', prop);
            }
        });
        
        debug('info', 'Alle Inline-Styles entfernt - CSS hat jetzt Vorrang');
    }, 10);
    
    b.addEventListener('click', () => {
      ensureModal();
      const m = document.getElementById('wp-c2pa-combined-modal');
      const summary = m.querySelector('[data-panel="summary"]');
      const raw = m.querySelector('.wp-c2pa-combined-raw');
      const src = chooseBestUrl(el);
      
      // Prüfe ob erweiterte Ansicht benötigt wird
      const hasIngredients = man && man.ingredients && Array.isArray(man.ingredients) && man.ingredients.length > 1;
      const hasActions = man && man.actions && Array.isArray(man.actions) && man.actions.length > 0;
      const needsAdvancedView = hasIngredients || hasActions;
      
      // Entferne zuerst alle bestehenden "Erweiterte Ansicht" Tabs und Panels
      const tabsContainer = m.querySelector('.wp-c2pa-combined-tabs');
      const existingAdvancedTab = tabsContainer.querySelector('[data-tab="advanced"]');
      const modalBody = m.querySelector('.wp-c2pa-combined-modal-body');
      const existingAdvancedPanel = modalBody.querySelector('[data-panel="advanced"]');
      
      if (existingAdvancedTab) {
        existingAdvancedTab.remove();
      }
      if (existingAdvancedPanel) {
        existingAdvancedPanel.remove();
      }
      
      // Füge "Erweiterte Ansicht" Tab hinzu wenn nötig
      if (needsAdvancedView) {
        // Füge Tab nach "Zusammenfassung" hinzu
        const summaryTab = tabsContainer.querySelector('[data-tab="summary"]');
        const advancedTab = document.createElement('button');
        advancedTab.className = 'wp-c2pa-combined-tab';
        advancedTab.setAttribute('data-tab', 'advanced');
        advancedTab.textContent = t('extendedView');
        summaryTab.parentNode.insertBefore(advancedTab, summaryTab.nextSibling);
        
        // Füge Panel hinzu
        const advancedPanel = document.createElement('div');
        advancedPanel.className = 'wp-c2pa-combined-tabpanel is-hidden';
        advancedPanel.setAttribute('data-panel', 'advanced');
        modalBody.appendChild(advancedPanel);
        
        // Event Listener für neuen Tab
        advancedTab.addEventListener('click', () => {
          m.querySelectorAll('.wp-c2pa-combined-tab').forEach(b => b.classList.toggle('is-active', b === advancedTab));
          m.querySelectorAll('.wp-c2pa-combined-tabpanel').forEach(p => p.classList.toggle('is-hidden', p.getAttribute('data-panel') !== 'advanced'));
        });
      }
      
      summary.innerHTML = renderSummary(man, meta, src, el);
      
      // Rendere erweiterte Ansicht wenn nötig
      if (needsAdvancedView) {
        const advancedPanel = m.querySelector('[data-panel="advanced"]');
        if (advancedPanel) {
          advancedPanel.innerHTML = renderAdvancedView(man, meta);
        }
      }
      
      if(raw) raw.textContent = safeJSON({ manifest:man, metadata:meta },2);
      
      // Stelle sicher, dass "Zusammenfassung" Tab immer aktiv ist
      const summaryTab = m.querySelector('[data-tab="summary"]');
      const summaryPanel = m.querySelector('[data-panel="summary"]');
      
      // Entferne alle aktiven Klassen
      m.querySelectorAll('.wp-c2pa-combined-tab').forEach(tab => tab.classList.remove('is-active'));
      m.querySelectorAll('.wp-c2pa-combined-tabpanel').forEach(panel => panel.classList.add('is-hidden'));
      
      // Aktiviere "Zusammenfassung" Tab
      if (summaryTab) {
        summaryTab.classList.add('is-active');
      }
      if (summaryPanel) {
        summaryPanel.classList.remove('is-hidden');
      } 
      
      // Behandle lange Texte nach dem Rendern
      setTimeout(() => {
        const longTexts = m.querySelectorAll('.wp-c2pa-combined-value');
        longTexts.forEach(textEl => {
          if (textEl.textContent.length > 50) {
            textEl.classList.add('long-text');
          }
        });
      }, 100);
      
      showModal();
    });
    
    // Füge Badge direkt nach dem Bild ein
    el.parentNode.insertBefore(b, el.nextSibling);
    
    // NEUE LÖSUNG: Registriere Badge global
    const badgeId = generateBadgeId(el);
    cfg.attachedBadges.add(badgeId);
    debug('info', 'Badge erfolgreich angehängt und global registriert', badgeId);
  }
  
  // STANDARD METHODE: Container wird erstellt
  function attachBadgeStandard(el, man, meta) {
    // Prüfe ob es sich um ein Elementor-Bildformat handelt
    if (isElementorImageFormat(el)) {
      // Für Elementor-Bildformate verwende die minimale Methode
      attachBadgeMinimal(el, man, meta);
      return;
    }
    
    // Erstelle einen unsichtbaren Wrapper nur für die Badge-Positionierung
    // Das Bild selbst bleibt unverändert
    let badgeContainer = el.parentElement;
    
    // Prüfe ob bereits ein Badge-Container existiert
    if (!badgeContainer || !badgeContainer.classList.contains('wp-c2pa-combined-badge-container')) {
      // Erstelle einen unsichtbaren Container für den Badge
      badgeContainer = document.createElement('div');
      badgeContainer.className = 'wp-c2pa-combined-badge-container';
      badgeContainer.style.position = 'relative';
      badgeContainer.style.display = 'inline-block';
      badgeContainer.style.width = 'fit-content';
      badgeContainer.style.height = 'fit-content';
      
      // Füge Container vor dem Bild ein und verschiebe das Bild in den Container
      el.parentNode.insertBefore(badgeContainer, el);
      badgeContainer.appendChild(el);
    }
    
    // Prüfe ob bereits ein Badge im Container existiert
    const existingBadge = badgeContainer.querySelector('.wp-c2pa-combined-badge');
    if (existingBadge) {
      debug('info', 'Badge bereits im Container vorhanden, überspringe');
      return;
    }
    
    // Erstelle Badge direkt hier (ohne createBadge Funktion)
    const b = document.createElement('button'); 
    b.type = 'button'; 
    b.className = 'wp-c2pa-combined-badge'; 
    
    // Positionierung basierend auf Einstellung
    const position = cfg.settings.badgePosition || 'bottom-left';
    b.style.position = 'absolute';
    b.style.zIndex = '10';
    
    // Proportionale Badge-Positionierung basierend auf Bildgröße
    const imageRect = el.getBoundingClientRect();
    const imageWidth = imageRect.width;
    const imageHeight = imageRect.height;
    
    // Berechne proportionale Abstände (5% der Bildgröße, mindestens 8px, maximal 20px)
    const marginX = Math.max(8, Math.min(20, Math.round(imageWidth * 0.05)));
    const marginY = Math.max(8, Math.min(20, Math.round(imageHeight * 0.05)));
    
    switch(position) {
      case 'bottom-left':
        b.style.left = marginX + 'px';
        b.style.bottom = marginY + 'px';
        break;
      case 'bottom-right':
        b.style.right = marginX + 'px';
        b.style.bottom = marginY + 'px';
        break;
      case 'top-left':
        b.style.left = marginX + 'px';
        b.style.top = marginY + 'px';
        break;
      case 'top-right':
        b.style.right = marginX + 'px';
        b.style.top = marginY + 'px';
        break;
    }
    
    // Apply custom badge styling
    const iconMap = {
      'shield': '🛡️',
      'check': '✅',
      'lock': '🔒',
      'star': '⭐',
      'verified': '✓',
      'certificate': '📜',
      'eye': '👁️',
      'info': 'ℹ️',
      'badge': '🏷️',
      'security': '🔐',
      'eye-shield': '<img src="' + pluginUrl + 'assets/images/eye_shield_badge_transparent.svg" style="width: 16px; height: 16px; vertical-align: middle;" alt="Eye Shield">'
    };
    
    const badgeColor = cfg.settings.badgeColor || '#0073aa';
    const badgeIcon = cfg.settings.badgeIcon || 'shield';
    const badgeText = cfg.settings.badgeText || 'Content Credentials';
    const badgeSize = cfg.settings.badgeSize || 'medium';
    const badgeStyle = cfg.settings.badgeStyle || 'solid';
    const badgeDisplay = cfg.settings.badgeDisplay || 'icon_text';
    // const badgeOpacity = cfg.settings.badgeOpacity || '1.0'; // DEAKTIVIERT - Badge ist immer 90% transparent
    const badgeShadow = cfg.settings.badgeShadow || 'light';
    const badgeBorderRadius = cfg.settings.badgeBorderRadius || '8';
    const badgePadding = cfg.settings.badgePadding || '8';
    const badgeFontSize = cfg.settings.badgeFontSize || '14';
    const badgeFontWeight = cfg.settings.badgeFontWeight || '600';
    
    // Add size and style classes - CSS hat Vorrang vor Inline-Styles
    b.classList.add('badge-size-' + badgeSize);
    b.classList.add('badge-style-' + badgeStyle);
    
    // KEINE Transparenz-Logik mehr - Badge ist immer 90% transparent
    // b.classList.add('transparent'); // DEAKTIVIERT
    debug('info', 'Badge wird immer mit 90% Transparenz angezeigt');
    
    // KEINE Inline-Styles mehr - CSS-Klassen übernehmen die Formatierung
    
    if(!man) b.classList.add('no-credentials'); 
    
    // Set badge content based on display setting
    let iconSymbol = iconMap[badgeIcon] || '🛡️';
    
    // Prüfe ob es ein Custom PNG Badge ist
    if (badgeIcon === 'custom-png') {
        const customBadgeUrl = cfg.settings.pluginUrl + 'assets/images/custom-badge.png';
        iconSymbol = '<img src="' + customBadgeUrl + '" style="width: 16px; height: 16px; vertical-align: middle;" alt="Custom Badge">';
    }
    
    // Prüfe ob es ein Eye Shield Badge ist
    if (badgeIcon === 'eye-shield') {
        iconSymbol = iconMap[badgeIcon];
    }
    
    switch(badgeDisplay) {
        case 'icon_only':
            b.innerHTML = iconSymbol;
            // CSS-Klassen übernehmen die Formatierung
            break;
        case 'text_only':
            b.innerHTML = badgeText;
            // CSS-Klassen übernehmen die Formatierung
            break;
        case 'icon_text':
        default:
            b.innerHTML = iconSymbol + ' ' + badgeText;
            // CSS-Klassen übernehmen die Formatierung
            break;
    }
    
    // Apply custom styling - Schriftfarbe setzen (wird später durch CSS überschrieben)
    // b.style.color = badgeColor; // DEAKTIVIERT - CSS hat Vorrang
    
    // KEINE Inline-Styles mehr - CSS-Klassen übernehmen die Formatierung
    
    // RADIKALE LÖSUNG: Entferne ALLE Inline-Styles die CSS überschreiben könnten
    setTimeout(() => {
        // CSS-Klassen haben Vorrang vor Inline-Styles
        if (b.style.fontSize) b.style.removeProperty('font-size');
        if (b.style.padding) b.style.removeProperty('padding');
        if (b.style.background) b.style.removeProperty('background');
        if (b.style.backgroundColor) b.style.removeProperty('background-color');
        if (b.style.border) b.style.removeProperty('border');
        if (b.style.color) b.style.removeProperty('color');
        
        // Zusätzliche Sicherheit: Entferne alle möglichen CSS-Eigenschaften
        const propertiesToRemove = ['background', 'background-color', 'border', 'color', 'font-size', 'padding'];
        propertiesToRemove.forEach(prop => {
            if (b.style[prop]) {
                b.style.removeProperty(prop);
                debug('info', 'Inline-Style entfernt', prop);
            }
        });
        
        debug('info', 'Alle Inline-Styles entfernt - CSS hat jetzt Vorrang');
    }, 10);
    
    // Apply new styling options (nur wenn nicht durch CSS überschrieben)
    // b.style.opacity = badgeOpacity; // DEAKTIVIERT - CSS übernimmt Transparenz
    b.style.borderRadius = badgeBorderRadius + 'px';
    b.style.fontWeight = badgeFontWeight;
    
    // Apply shadow effects
    switch(badgeShadow) {
      case 'none':
        b.style.boxShadow = 'none';
        break;
      case 'light':
        b.style.boxShadow = '0 2px 4px rgba(0,0,0,0.1)';
        break;
      case 'medium':
        b.style.boxShadow = '0 4px 8px rgba(0,0,0,0.15)';
        break;
      case 'strong':
        b.style.boxShadow = '0 8px 16px rgba(0,0,0,0.2)';
        break;
    }
    
    b.addEventListener('click',()=>{ 
      ensureModal(); 
      const m=document.getElementById('wp-c2pa-combined-modal'); 
      const summary=m.querySelector('[data-panel="summary"]'); 
      const raw=m.querySelector('.wp-c2pa-combined-raw'); 
      const src=chooseBestUrl(el); 
      
      // Prüfe ob erweiterte Ansicht benötigt wird
      const hasIngredients = man && man.ingredients && Array.isArray(man.ingredients) && man.ingredients.length > 1;
      const hasActions = man && man.actions && Array.isArray(man.actions) && man.actions.length > 0;
      const needsAdvancedView = hasIngredients || hasActions;
      
      // Entferne zuerst alle bestehenden "Erweiterte Ansicht" Tabs und Panels
      const tabsContainer = m.querySelector('.wp-c2pa-combined-tabs');
      const existingAdvancedTab = tabsContainer.querySelector('[data-tab="advanced"]');
      const modalBody = m.querySelector('.wp-c2pa-combined-modal-body');
      const existingAdvancedPanel = modalBody.querySelector('[data-panel="advanced"]');
      
      if (existingAdvancedTab) {
        existingAdvancedTab.remove();
      }
      if (existingAdvancedPanel) {
        existingAdvancedPanel.remove();
      }
      
      // Füge "Erweiterte Ansicht" Tab hinzu wenn nötig
      if (needsAdvancedView) {
        // Füge Tab nach "Zusammenfassung" hinzu
        const summaryTab = tabsContainer.querySelector('[data-tab="summary"]');
        const advancedTab = document.createElement('button');
        advancedTab.className = 'wp-c2pa-combined-tab';
        advancedTab.setAttribute('data-tab', 'advanced');
        advancedTab.textContent = t('extendedView');
        summaryTab.parentNode.insertBefore(advancedTab, summaryTab.nextSibling);
        
        // Füge Panel hinzu
        const advancedPanel = document.createElement('div');
        advancedPanel.className = 'wp-c2pa-combined-tabpanel is-hidden';
        advancedPanel.setAttribute('data-panel', 'advanced');
        modalBody.appendChild(advancedPanel);
        
        // Event Listener für neuen Tab
        advancedTab.addEventListener('click', () => {
          m.querySelectorAll('.wp-c2pa-combined-tab').forEach(b => b.classList.toggle('is-active', b === advancedTab));
          m.querySelectorAll('.wp-c2pa-combined-tabpanel').forEach(p => p.classList.toggle('is-hidden', p.getAttribute('data-panel') !== 'advanced'));
        });
      }
      
      summary.innerHTML=renderSummary(man, meta, src, el); 
      
      // Rendere erweiterte Ansicht wenn nötig
      if (needsAdvancedView) {
        const advancedPanel = m.querySelector('[data-panel="advanced"]');
        if (advancedPanel) {
          advancedPanel.innerHTML = renderAdvancedView(man, meta);
        }
      }
      
      if(raw) raw.textContent=safeJSON({ manifest:man, metadata:meta },2);
      
      // Stelle sicher, dass "Zusammenfassung" Tab immer aktiv ist
      const summaryTab = m.querySelector('[data-tab="summary"]');
      const summaryPanel = m.querySelector('[data-panel="summary"]');
      
      // Entferne alle aktiven Klassen
      m.querySelectorAll('.wp-c2pa-combined-tab').forEach(tab => tab.classList.remove('is-active'));
      m.querySelectorAll('.wp-c2pa-combined-tabpanel').forEach(panel => panel.classList.add('is-hidden'));
      
      // Aktiviere "Zusammenfassung" Tab
      if (summaryTab) {
        summaryTab.classList.add('is-active');
      }
      if (summaryPanel) {
        summaryPanel.classList.remove('is-hidden');
      } 
      
      // Behandle lange Texte nach dem Rendern
      setTimeout(() => {
        const longTexts = m.querySelectorAll('.wp-c2pa-combined-value');
        longTexts.forEach(textEl => {
          if (textEl.textContent.length > 50) {
            textEl.classList.add('long-text');
          }
        });
      }, 100);
      
      showModal(); 
    }); 
    
    // Füge Badge in den Container ein (über dem Bild positioniert)
    badgeContainer.appendChild(b);
    
    // NEUE LÖSUNG: Registriere Badge global
    const badgeId = generateBadgeId(el);
    cfg.attachedBadges.add(badgeId);
    debug('info', 'Badge erfolgreich angehängt und global registriert', badgeId);
  }
  


  async function processEl(el){ 
    debug('info', 'processEl gestartet', {
      src: el.src || el.dataset.src,
      className: el.className,
      tagName: el.tagName,
      alreadyProcessed: el.dataset.wpC2paCombinedProcessed
    });
    
    if(el.dataset.wpC2paCombinedProcessed) {
      debug('info', 'Bereits verarbeitet, überspringe');
      return; 
    }
    el.dataset.wpC2paCombinedProcessed='1'; 
    
    // Überspringe Videos - Content Credentials nur für Bilder
    if (el.tagName.toLowerCase() === 'video') {
      debug('info', 'Video erkannt, überspringe Content Credentials');
      return;
    }
    
    // Prüfe ob das Element sichtbar ist, um Flackern zu vermeiden
    if (el.offsetParent === null && !el.getBoundingClientRect().width) {
      debug('info', 'Element nicht sichtbar, überspringe');
      return; // Element ist nicht sichtbar
    }
    
    let loadingElement = null;
    if(cfg.settings.showLoading){ 
      loadingElement = document.createElement('div'); 
      loadingElement.className='wp-c2pa-combined-loading'; 
      loadingElement.textContent=cfg.settings.strings?.loading||t('loading'); 
      el.parentNode&&el.parentNode.appendChild(loadingElement); 
    }
    
    try {
      const [man, meta]=await Promise.all([ readManifest(el), readMetadata(el) ]);
      
      // Entferne Loading-Indikator
      if(loadingElement && loadingElement.parentNode) {
        loadingElement.remove();
      }
      
          // Füge Badge nur hinzu, wenn das Element noch existiert
    if (el.parentNode) {
      attachBadge(el, man, meta);
      
      // Markiere das Element als "Badge angehängt" um mehrfache Verarbeitung zu verhindern
      if (el.dataset.wpC2paCombinedBadgeAttached) {
        debug('info', 'Badge erfolgreich angehängt');
      }
      
      // Benachrichtigungen nur bei sichtbaren Elementen und wenn nicht deaktiviert
      // UND NICHT auf der Startseite wenn ausgeschlossen
      if ((el.offsetParent !== null || el.getBoundingClientRect().width > 0) && 
          !cfg.settings.disableNotifications && 
          !(cfg.settings.excludeHomepage && isHomepage())) {
        if(man) notify(t('contentCredentialsFound'),'success'); 
        else if(meta&&meta.summary&&Object.keys(meta.summary).length) notify(t('metadataFound'),'info'); 
        else if(cfg.settings.showNoCredentialsMessage) notify(t('noCredentialsFound'),'warning');
      }
    }
    } catch (error) {
      // Entferne Loading-Indikator bei Fehlern
      if(loadingElement && loadingElement.parentNode) {
        loadingElement.remove();
      }
      debug('error', 'Fehler beim Verarbeiten des Bildes', error);
    }
  }

  function scan(){ 
    debug('info', 'Scan gestartet');
    
    // NEUE LÖSUNG: Lösche alle bestehenden Badges vor dem Scan
    cleanupExistingBadges();
    
    // Wähle Scan-Methode basierend auf Theme-Kompatibilitätsmodus
    const positioningMethod = cfg.settings.themeCompatibilityMode || 'standard';
    debug('info', 'Aktueller Modus', positioningMethod);
    
    if (positioningMethod === 'minimal') {
      // MINIMAL MODUS: Exakte Logik von Version 003205
      debug('info', 'Minimal-Modus - Suche nach img');
      const allImages = Array.from(document.querySelectorAll('img'));
      debug('info', 'Gefundene Bilder', allImages.length);
      
      const nodes = allImages.filter(n => !n.dataset.wpC2paCombinedProcessed);
      debug('info', 'Neue Bilder zum Verarbeiten', nodes.length);
      nodes.forEach(n => processEl(n));
    } else {
      // STANDARD MODUS: Erweiterte Suche für Theme-Bilder
      debug('info', 'Standard-Modus - Erweiterte Suche');
      const selectors = [
        'img',
        'picture img',
        'figure img',
        '.wp-block-image img',
        '.gallery img',
        '.wp-block-gallery img',
        '.blocks-gallery-grid img',
        '.blocks-gallery-item img',
        '.gallery-item img',
        '.attachment img',
        '[class*="gallery"] img',
        '[class*="attachment"] img',
        '[class*="image"] img',
        '[class*="photo"] img',
        '[class*="media"] img',
        '[class*="thumb"] img',
        '[class*="preview"] img',
        '[class*="featured"] img',
        '[class*="hero"] img',
        '[class*="banner"] img',
        '[class*="slider"] img',
        '[class*="carousel"] img'
      ];
      
      const allNodes = new Set();
      selectors.forEach(selector => {
        try {
          const found = document.querySelectorAll(selector);
          debug('info', `Selector "${selector}" gefunden`, found.length);
          found.forEach(node => {
                      if (node.tagName === 'IMG') {
            allNodes.add(node);
          }
          });
        } catch (e) {
          debug('warning', `Selector "${selector}" fehlgeschlagen`, e.message);
        }
      });
      
      const nodes = Array.from(allNodes).filter(n => !n.dataset.wpC2paCombinedProcessed);
      debug('info', 'Neue Bilder zum Verarbeiten', nodes.length);
      nodes.forEach(n => processEl(n));
    }
  }

  async function init(){ 
    await loadC2PA(); 
    if(cfg.settings.autoScan!==false) scan(); 
    
    // NEUE LÖSUNG: Keine MutationObserver mehr - nur manuelle Scans
    // Verzögerte Scans für dynamisch geladene Bilder
    setTimeout(() => scan(), 2000);
    setTimeout(() => scan(), 5000);
    setTimeout(() => scan(), 10000);
    
    // MutationObserver deaktiviert - verursacht zu viele Duplikate
    debug('info', 'MutationObserver deaktiviert - nur manuelle Scans aktiv');
  }

  if(document.readyState==='loading') document.addEventListener('DOMContentLoaded',init); 
  else init();

  // Public API
  window.WPC2PACombinedViewer={ 
    scan, 
    processEl,
    // Manueller Scan für Theme-Bilder
    scanThemeImages: () => {
      debug('info', 'Manueller Theme-Bild-Scan gestartet');
      scan();
    }
  };
})();
