<?php
/**
 * Plugin Name: C2PA Content Credentials Viewer
 * Description: Truth is the best branding. Show what's real, because transparency builds trust. This plugin displays C2PA Content Credentials from images and enables detailed analysis of image authenticity with extended metadata and editing history. Based on <a href="https://contentcredentials.org/">Content Credentials</a>.
 * Version: 4.9.0
 * Author: <a href="https://www.thomasbrandt.eu">Thomas Brandt</a>
 * License: GPLv2 or later
 * Text Domain: wp-c2pa-viewer-combined
 * Requires at least: 5.0
 * Tested up to: 6.4
 * Requires PHP: 7.4
 */

// Sicherheitscheck
if (!defined('ABSPATH')) {
    exit;
}

// Plugin-Konstanten
if (!defined('WP_C2PA_COMBINED_VERSION')) {
    define('WP_C2PA_COMBINED_VERSION', '5.4.8-FINAL');
}

if (!defined('WP_C2PA_COMBINED_URL')) {
    define('WP_C2PA_COMBINED_URL', plugin_dir_url(__FILE__));
}

if (!defined('WP_C2PA_COMBINED_PATH')) {
    define('WP_C2PA_COMBINED_PATH', plugin_dir_path(__FILE__));
}

if (!defined('WP_C2PA_COMBINED_BASENAME')) {
    define('WP_C2PA_COMBINED_BASENAME', plugin_basename(__FILE__));
}

/**
 * Hauptklasse für das kombinierte C2PA Viewer Plugin
 */
class WP_C2PA_Viewer_Combined {
    
    /**
     * Plugin-Initialisierung
     */
    public function __construct() {
        add_action('wp_enqueue_scripts', [$this, 'enqueue_scripts']);
        add_action('wp_head', [$this, 'add_meta_tags']);
        add_action('admin_menu', [$this, 'add_admin_menu']);
        add_action('admin_init', [$this, 'register_settings']);
        add_action('admin_enqueue_scripts', [$this, 'admin_enqueue_scripts']);
        
        // Bild-Skalierung deaktivieren
        add_filter('big_image_size_threshold', [$this, 'disable_image_scaling'], 10, 4);
        
        // Shortcode für manuelle Einbindung
        add_shortcode('c2pa_viewer_combined', [$this, 'shortcode_handler']);
        
        // Aktivierung und Deaktivierung
        register_activation_hook(__FILE__, [$this, 'activate']);
        register_deactivation_hook(__FILE__, [$this, 'deactivate']);
    }
    
    /**
     * Scripts und Styles laden
     */
    public function enqueue_scripts() {
        // Nur laden wenn nötig
        if (!$this->should_load_assets()) {
            return;
        }
        
        // CSS laden
        wp_enqueue_style(
            'wp-c2pa-combined',
            WP_C2PA_COMBINED_URL . 'assets/css/c2pa-viewer-combined.css',
            [],
            WP_C2PA_COMBINED_VERSION
        );
        
        // Benachrichtigungen CSS laden
        wp_enqueue_style(
            'wp-c2pa-combined-notifications',
            WP_C2PA_COMBINED_URL . 'assets/css/notifications.css',
            ['wp-c2pa-combined'],
            WP_C2PA_COMBINED_VERSION
        );
        
        // JavaScript laden
        wp_enqueue_script(
            'wp-c2pa-combined',
            WP_C2PA_COMBINED_URL . 'assets/js/c2pa-viewer-combined.js',
            [],
            WP_C2PA_COMBINED_VERSION,
            true
        );
        
        // Konfiguration an JS übergeben
        wp_localize_script('wp-c2pa-combined', 'WPC2PACombinedConfig', [
            'version' => WP_C2PA_COMBINED_VERSION,
            'language' => $this->get_option('language', 'de'),
            'pluginUrl' => WP_C2PA_COMBINED_URL,
            'c2paVersion' => $this->get_option('c2pa_version', '0.30.11'),
            'autoScan' => $this->get_option('auto_scan', true),
            'badgePosition' => $this->get_option('badge_position', 'bottom-left'),
            'showLoading' => $this->get_option('show_loading', true),
            'showComparison' => $this->get_option('show_comparison', true),
            'showRawData' => $this->get_option('show_raw_data', false),
            'badgeDisplay' => $this->get_option('badge_display', 'icon_text'),
            'badgeColor' => $this->get_option('badge_color', '#0073aa'),
            'badgeIcon' => $this->get_option('badge_icon', 'shield'),
            'badgeText' => $this->get_option('badge_text', 'Content Credentials'),
            'badgeSize' => $this->get_option('badge_size', 'medium'),
            'badgeStyle' => $this->get_option('badge_style', 'solid'),
            // 'badgeOpacity' => $this->get_option('badge_opacity', '1.0'), // DEAKTIVIERT - Badge ist immer 90% transparent
            'badgeShadow' => $this->get_option('badge_shadow', 'light'),
            'badgeBorderRadius' => $this->get_option('badge_border_radius', '8'),
            'badgePadding' => $this->get_option('badge_padding', '8'),
            'badgeFontSize' => $this->get_option('badge_font_size', '14'),
            'badgeFontWeight' => $this->get_option('badge_font_weight', '600'),
            'excludeFeaturedImages' => $this->get_option('exclude_featured_images', false),
            'excludeHomepage' => $this->get_option('exclude_homepage', false),
            'themeCompatibilityMode' => $this->get_option('theme_compatibility_mode', 'standard'),
            'ajaxUrl' => admin_url('admin-ajax.php'),
            'nonce' => wp_create_nonce('wp_c2pa_combined_nonce'),
            'strings' => [
                'loading' => __('Lade Content Credentials...', 'wp-c2pa-viewer-combined'),
                'noCredentials' => __('Keine Content Credentials gefunden', 'wp-c2pa-viewer-combined'),
                'error' => __('Fehler beim Laden der Credentials', 'wp-c2pa-viewer-combined'),
                'close' => __('Schließen', 'wp-c2pa-viewer-combined'),
                'basicData' => __('Basis C2PA-Daten', 'wp-c2pa-viewer-combined'),
                'extendedData' => __('Erweiterte Analyse', 'wp-c2pa-viewer-combined'),
                'rawData' => __('Rohdaten', 'wp-c2pa-viewer-combined'),
                'contentCredentials' => __('Content Credentials', 'wp-c2pa-viewer-combined'),
                'basicTabDescription' => __('Nur C2PA-Manifest-Daten (wie gold5)', 'wp-c2pa-viewer-combined'),
                'extendedTabDescription' => __('C2PA + Metadaten + AI-Erkennung (wie v1.1.2)', 'wp-c2pa-viewer-combined'),
            ]
        ]);
    }
    
    /**
     * Meta-Tags für SEO hinzufügen
     */
    public function add_meta_tags() {
        echo '<meta name="c2pa-plugin" content="WP C2PA Viewer Combined ' . WP_C2PA_COMBINED_VERSION . '">' . "\n";
    }
    
    /**
     * Admin-Menü hinzufügen
     */
    public function add_admin_menu() {
        add_menu_page(
            'C2PA Viewer Combined',
            'C2PA Viewer',
            'manage_options',
            'wp-c2pa-viewer-combined',
            [$this, 'admin_page'],
            $this->get_shield_icon_svg(),
            30
        );
    }
    
    /**
     * Admin Scripts und Styles laden
     */
    public function admin_enqueue_scripts($hook) {
        if ($hook !== 'toplevel_page_wp-c2pa-viewer-combined') {
            return;
        }
        
        wp_enqueue_style('wp-color-picker');
        wp_enqueue_script('wp-color-picker');
        
        wp_enqueue_style(
            'wp-c2pa-combined-admin',
            WP_C2PA_COMBINED_URL . 'assets/css/admin.css',
            [],
            WP_C2PA_COMBINED_VERSION
        );
        
        wp_enqueue_script(
            'wp-c2pa-combined-admin',
            WP_C2PA_COMBINED_URL . 'assets/js/admin.js',
            ['jquery', 'wp-color-picker'],
            WP_C2PA_COMBINED_VERSION,
            true
        );
        
        // Plugin-URL für Admin-JavaScript
        wp_localize_script('wp-c2pa-combined-admin', 'pluginUrl', WP_C2PA_COMBINED_URL);
    }
    
    /**
     * Schild-Icon als SVG für das Menü
     */
    private function get_shield_icon_svg() {
        return 'data:image/svg+xml;base64,' . base64_encode('
            <svg width="20" height="20" viewBox="0 0 24 24" fill="none" xmlns="http://www.w3.org/2000/svg">
                <path d="M12 2L3 7V11C3 16.55 6.84 21.74 12 23C17.16 21.74 21 16.55 21 11V7L12 2ZM12 7C13.1 7 14 7.9 14 9C14 10.1 13.1 11 12 11C10.9 11 10 10.1 10 9C10 7.9 10.9 7 12 7ZM12 17C10.5 17 9.14 16.38 8.13 15.37L9.55 13.95C10.18 14.58 11.05 15 12 15C12.95 15 13.82 14.58 14.45 13.95L15.87 15.37C14.86 16.38 13.5 17 12 17Z" fill="currentColor"/>
            </svg>
        ');
    }
    
    /**
     * Einstellungen registrieren
     */
    public function register_settings() {
        register_setting('wp_c2pa_combined_options', 'wp_c2pa_combined_settings');
        
        // Spracheinstellungen
        add_settings_section(
            'wp_c2pa_combined_language',
            $this->get_section_title('language'),
            [$this, 'language_section_callback'],
            'wp-c2pa-viewer-combined'
        );
        
        // Haupteinstellungen
        add_settings_section(
            'wp_c2pa_combined_main',
            $this->get_section_title('main'),
            [$this, 'settings_section_callback'],
            'wp-c2pa-viewer-combined'
        );
        
        // Badge Customization
        add_settings_section(
            'wp_c2pa_combined_badge',
            $this->get_section_title('badge_customization'),
            [$this, 'badge_section_callback'],
            'wp-c2pa-viewer-combined'
        );
        
        add_settings_section(
            'wp_c2pa_combined_pages',
            $this->get_section_title('page_settings'),
            [$this, 'pages_section_callback'],
            'wp-c2pa-viewer-combined'
        );
        

        
        // Spracheinstellungen Felder
        add_settings_field('language', $this->get_field_title('language'), [$this, 'language_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_language');
        
        // Haupteinstellungen Felder
        add_settings_field('c2pa_version', $this->get_field_title('c2pa_version'), [$this, 'version_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_main');
        add_settings_field('auto_scan', $this->get_field_title('auto_scan'), [$this, 'checkbox_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_main', ['field' => 'auto_scan', 'label' => 'Bilder automatisch nach C2PA-Inhalten scannen']);
        add_settings_field('show_comparison', $this->get_field_title('show_comparison'), [$this, 'checkbox_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_main', ['field' => 'show_comparison', 'label' => 'Vergleich zwischen Basis- und erweiterten Daten anzeigen']);
        add_settings_field('show_raw_data', $this->get_field_title('show_raw_data'), [$this, 'checkbox_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_main', ['field' => 'show_raw_data', 'label' => 'Rohdaten-Tab in der Bildanalyse anzeigen']);
        
        // Badge Felder
        add_settings_field('badge_display', $this->get_field_title('badge_display'), [$this, 'select_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', [
            'field' => 'badge_display',
            'options' => [
                'icon_only' => 'Nur Symbol',
                'text_only' => 'Nur Text',
                'icon_text' => 'Symbol und Text'
            ]
        ]);
        add_settings_field('badge_text', $this->get_field_title('badge_text'), [$this, 'conditional_text_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', ['field' => 'badge_text', 'default' => 'Content Credentials', 'description' => 'Text der auf dem Badge angezeigt wird (wird nur bei "Nur Text" oder "Symbol und Text" verwendet)']);
        add_settings_field('badge_position', $this->get_field_title('badge_position'), [$this, 'select_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', [
            'field' => 'badge_position',
            'options' => [
                'bottom-left' => 'Unten links',
                'bottom-right' => 'Unten rechts', 
                'top-left' => 'Oben links',
                'top-right' => 'Oben rechts'
            ]
        ]);
        add_settings_field('badge_color', $this->get_field_title('badge_color'), [$this, 'color_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', ['field' => 'badge_color', 'default' => '#0073aa']);
        add_settings_field('badge_icon', $this->get_field_title('badge_icon'), [$this, 'icon_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', ['field' => 'badge_icon']);
        add_settings_field('badge_size', $this->get_field_title('badge_size'), [$this, 'select_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', [
            'field' => 'badge_size',
            'options' => [
                'small' => 'Klein',
                'medium' => 'Mittel',
                'large' => 'Groß'
            ]
        ]);
        add_settings_field('badge_style', $this->get_field_title('badge_style'), [$this, 'select_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', [
            'field' => 'badge_style',
            'options' => [
                'solid' => 'Ausgefüllt',
                'outline' => 'Rahmen',
                'minimal' => 'Minimal'
            ]
        ]);
        
        // Erweiterte Badge-Optionen
        // Transparenz-Einstellung entfernt - Badge ist immer 90% transparent
        // add_settings_field('badge_opacity', $this->get_field_title('badge_opacity'), [$this, 'select_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', [
        //     'field' => 'badge_opacity',
        //     'options' => [
        //         '0.7' => '70% (transparent)',
        //         '0.8' => '80%',
        //         '0.9' => '90%',
        //         '1.0' => '100% (undurchsichtig)'
        //     ]
        // ]);
        add_settings_field('badge_shadow', $this->get_field_title('badge_shadow'), [$this, 'select_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', [
            'field' => 'badge_shadow',
            'options' => [
                'none' => 'Kein Schatten',
                'light' => 'Leichter Schatten',
                'medium' => 'Mittlerer Schatten',
                'strong' => 'Starker Schatten'
            ]
        ]);

        add_settings_field('badge_border_radius', $this->get_field_title('badge_border_radius'), [$this, 'select_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', [
            'field' => 'badge_border_radius',
            'options' => [
                '0' => 'Eckig',
                '4' => 'Leicht gerundet',
                '8' => 'Gerundet',
                '12' => 'Stark gerundet',
                '20' => 'Sehr rund'
            ]
        ]);
        add_settings_field('badge_padding', $this->get_field_title('badge_padding'), [$this, 'select_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', [
            'field' => 'badge_padding',
            'options' => [
                '4' => 'Kompakt (4px)',
                '6' => 'Klein (6px)',
                '8' => 'Standard (8px)',
                '12' => 'Groß (12px)',
                '16' => 'Sehr groß (16px)'
            ]
        ]);
        add_settings_field('badge_font_size', $this->get_field_title('badge_font_size'), [$this, 'select_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', [
            'field' => 'badge_font_size',
            'options' => [
                '10' => 'Sehr klein (10px)',
                '12' => 'Klein (12px)',
                '14' => 'Standard (14px)',
                '16' => 'Groß (16px)',
                '18' => 'Sehr groß (18px)'
            ]
        ]);
        add_settings_field('badge_font_weight', $this->get_field_title('badge_font_weight'), [$this, 'select_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_badge', [
            'field' => 'badge_font_weight',
            'options' => [
                'normal' => 'Normal',
                '500' => 'Mittel',
                '600' => 'Semi-Bold',
                '700' => 'Bold',
                '800' => 'Extra Bold'
            ]
        ]);
        
        // Seiten-Einstellungen Felder
        add_settings_field('exclude_pages', $this->get_field_title('exclude_pages'), [$this, 'textarea_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_pages', [
            'field' => 'exclude_pages', 
            'description' => 'Eine URL pro Zeile. Badges werden auf diesen Seiten nicht angezeigt. Beispiel: /kontakt/, /impressum/'
        ]);
        add_settings_field('exclude_post_types', $this->get_field_title('exclude_post_types'), [$this, 'checkbox_list_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_pages', [
            'field' => 'exclude_post_types',
            'options' => [
                'page' => 'Seiten',
                'post' => 'Beiträge',
                'attachment' => 'Medien',
                'product' => 'Produkte (WooCommerce)',
                'shop_order' => 'Bestellungen (WooCommerce)'
            ]
        ]);
        
        // Titelbilder-Option hinzufügen
        add_settings_field('exclude_featured_images', $this->get_field_title('exclude_featured_images'), [$this, 'checkbox_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_pages', [
            'field' => 'exclude_featured_images',
            'label' => 'Badges für Titelbilder von Seiten und Beiträgen deaktivieren'
        ]);
        
        // Benachrichtigungen-Option hinzufügen
        add_settings_field('disable_notifications', $this->get_field_title('disable_notifications'), [$this, 'checkbox_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_pages', [
            'field' => 'disable_notifications',
            'label' => 'Benachrichtigungen für Content Credentials und Metadaten deaktivieren'
        ]);
        
        // Theme-Kompatibilitäts-Option hinzufügen
        add_settings_field('theme_compatibility_mode', $this->get_field_title('theme_compatibility_mode'), [$this, 'select_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_pages', [
            'field' => 'theme_compatibility_mode',
            'options' => [
                'standard' => 'Standard (Gutenberg/Standard-Theme) - Badge schwebt über Bildern',
                'minimal' => 'Minimal (Theme-kompatibel) - Keine DOM-Manipulation, Badge neben Bildern'
            ]
        ]);
        
        // Startseite ausschließen Option hinzufügen
        add_settings_field('exclude_homepage', $this->get_field_title('exclude_homepage'), [$this, 'checkbox_field_callback'], 'wp-c2pa-viewer-combined', 'wp_c2pa_combined_pages', [
            'field' => 'exclude_homepage',
            'label' => $this->get_option('language', 'de') === 'en' ? 'Disable Content Credentials on homepage' : 'Content Credentials auf der Startseite deaktivieren'
        ]);
    }
    
    /**
     * Sektionstitel basierend auf Sprache
     */
    private function get_section_title($section) {
        $current_language = $this->get_option('language', 'de');
        $titles = [
            'language' => [
                'de' => '🌍 Sprache',
                'en' => '🌍 Language'
            ],
            'main_settings' => [
                'de' => 'Haupteinstellungen',
                'en' => 'Main Settings'
            ],
            'badge_customization' => [
                'de' => 'Badge Anpassung',
                'en' => 'Badge Customization'
            ],
            'page_settings' => [
                'de' => 'Seiten-Einstellungen',
                'en' => 'Page Settings'
            ]
        ];
        
        return isset($titles[$section][$current_language]) ? $titles[$section][$current_language] : $titles[$section]['de'];
    }
    
    /**
     * Feldtitel basierend auf Sprache
     */
    private function get_field_title($field) {
        $current_language = $this->get_option('language', 'de');
        $titles = [
            'language' => [
                'de' => '🇩🇪🇺🇸 Sprache',
                'en' => '🇩🇪🇺🇸 Language'
            ],
            'c2pa_version' => [
                'de' => 'C2PA Version',
                'en' => 'C2PA Version'
            ],
            'auto_scan' => [
                'de' => 'Automatischer Scan',
                'en' => 'Auto Scan'
            ],
            'show_comparison' => [
                'de' => 'Vergleich anzeigen',
                'en' => 'Show Comparison'
            ],
            'show_raw_data' => [
                'de' => 'Rohdaten anzeigen',
                'en' => 'Show Raw Data'
            ],
            'badge_display' => [
                'de' => 'Badge Anzeige',
                'en' => 'Badge Display'
            ],
            'badge_text' => [
                'de' => 'Badge Text',
                'en' => 'Badge Text'
            ],
            'badge_position' => [
                'de' => 'Badge Position',
                'en' => 'Badge Position'
            ],
            'badge_color' => [
                'de' => 'Schriftfarbe',
                'en' => 'Text Color'
            ],
            'badge_icon' => [
                'de' => 'Badge Symbol',
                'en' => 'Badge Icon'
            ],
            'badge_size' => [
                'de' => 'Badge Größe',
                'en' => 'Badge Size'
            ],
            'badge_style' => [
                'de' => 'Badge Stil',
                'en' => 'Badge Style'
            ],
                    // 'badge_opacity' => [
        //     'de' => 'Badge Transparenz',
        //     'en' => 'Badge Opacity'
        // ], // DEAKTIVIERT
            'badge_shadow' => [
                'de' => 'Badge Schatten',
                'en' => 'Badge Shadow'
            ],
            'badge_border_radius' => [
                'de' => 'Badge Rundung',
                'en' => 'Badge Border Radius'
            ],
            'badge_padding' => [
                'de' => 'Badge Innenabstand',
                'en' => 'Badge Padding'
            ],
            'badge_font_size' => [
                'de' => 'Badge Schriftgröße',
                'en' => 'Badge Font Size'
            ],
            'badge_font_weight' => [
                'de' => 'Badge Schriftstärke',
                'en' => 'Badge Font Weight'
            ],
            'exclude_pages' => [
                'de' => 'Seiten ausschließen',
                'en' => 'Exclude Pages'
            ],
            'exclude_post_types' => [
                'de' => 'Beitragstypen ausschließen',
                'en' => 'Exclude Post Types'
            ],
            'exclude_featured_images' => [
                'de' => 'Titelbilder ausschließen',
                'en' => 'Exclude Featured Images'
            ],
            'disable_notifications' => [
                'de' => 'Benachrichtigungen deaktivieren',
                'en' => 'Disable Notifications'
            ],
            'theme_compatibility_mode' => [
                'de' => 'Theme-Kompatibilitätsmodus',
                'en' => 'Theme Compatibility Mode'
            ],
            'exclude_homepage' => [
                'de' => 'Startseite ausschließen',
                'en' => 'Exclude Homepage'
            ]
        ];
        
        return isset($titles[$field][$current_language]) ? $titles[$field][$current_language] : $titles[$field]['de'];
    }
    
    /**
     * Spracheinstellungen Sektion Callback
     */
    public function language_section_callback() {
        $current_language = $this->get_option('language', 'de');
        if ($current_language === 'de') {
            echo '<p><strong>🇩🇪 Deutsch ausgewählt</strong></p>';
        } else {
            echo '<p><strong>🇺🇸 English selected</strong></p>';
        }
        
        // JavaScript für dynamische Anzeige des Hinweistextes
        echo '<script>
        document.addEventListener("DOMContentLoaded", function() {
            const languageSelect = document.getElementById("language-select");
            const currentLanguage = "' . $current_language . '";
            
            function updateSaveMessage() {
                const selectedLanguage = languageSelect.value;
                let saveMessage = document.getElementById("language-save-message");
                
                if (selectedLanguage !== currentLanguage) {
                    if (!saveMessage) {
                        saveMessage = document.createElement("p");
                        saveMessage.id = "language-save-message";
                        saveMessage.style.color = "#d63638";
                        saveMessage.style.fontWeight = "bold";
                        saveMessage.style.marginTop = "10px";
                    }
                    
                    if (selectedLanguage === "de") {
                        saveMessage.textContent = "Speichern, um die Sprache umzustellen.";
                    } else {
                        saveMessage.textContent = "Save to change the language.";
                    }
                    
                    if (!saveMessage.parentNode) {
                        languageSelect.parentNode.appendChild(saveMessage);
                    }
                } else {
                    if (saveMessage) {
                        saveMessage.remove();
                    }
                }
            }
            
            if (languageSelect) {
                languageSelect.addEventListener("change", updateSaveMessage);
                updateSaveMessage(); // Initial check
            }
        });
        </script>';
    }
    
    /**
     * Sprache Feld Callback
     */
    public function language_field_callback() {
        $value = $this->get_option('language', 'de');
        echo '<select name="wp_c2pa_combined_settings[language]" id="language-select">';
        echo '<option value="de" ' . selected('de', $value, false) . '>🇩🇪 Deutsch</option>';
        echo '<option value="en" ' . selected('en', $value, false) . '>🇺🇸 English</option>';
        echo '</select>';
        echo '<p class="description">🇩🇪 Wählen Sie Ihre bevorzugte Sprache für die Einstellungen und Benutzeroberfläche.</p>';
        echo '<p class="description">🇺🇸 Choose your preferred language for settings and user interface.</p>';
    }
    
    /**
     * Einstellungen Sektion Callback
     */
    public function settings_section_callback() {
        $current_language = $this->get_option('language', 'de');
        if ($current_language === 'de') {
            echo '<p>Konfigurieren Sie die Einstellungen für den kombinierten C2PA Viewer.</p>';
        } else {
            echo '<p>Configure the settings for the combined C2PA Viewer.</p>';
        }
    }
    
    /**
     * Badge Sektion Callback
     */
    public function badge_section_callback() {
        echo '<p>Passen Sie das Aussehen und Verhalten der Content Credentials Badges an.</p>';
        echo '<div id="badge-preview-section" style="margin: 20px 0; padding: 20px; border: 1px solid #ddd; background: #f9f9f9;">';
        echo '<h4>Live-Vorschau:</h4>';
        echo '<div style="position: relative; display: inline-block; background: #eee; padding: 20px;">';
        echo '<img src="data:image/svg+xml;base64,' . base64_encode('<svg width="200" height="150" xmlns="http://www.w3.org/2000/svg"><rect width="200" height="150" fill="#ddd"/><text x="100" y="75" text-anchor="middle" fill="#666">Beispielbild</text></svg>') . '" alt="Beispiel" style="display: block;">';
        echo '<div id="badge-preview" class="wp-c2pa-combined-badge">Content Credentials</div>';
        echo '</div>';
        echo '</div>';
    }
    
    /**
     * Seiten Sektion Callback
     */
    public function pages_section_callback() {
        echo '<p>Konfigurieren Sie auf welchen Seiten die Content Credentials Badges angezeigt werden sollen.</p>';
    }
    
    /**
     * Version Feld Callback
     */
    public function version_field_callback() {
        $value = $this->get_option('c2pa_version', '0.30.11');
        echo '<input type="text" name="wp_c2pa_combined_settings[c2pa_version]" value="' . esc_attr($value) . '" class="regular-text">';
        echo '<p class="description">Die Version der C2PA-Bibliothek, die verwendet werden soll.</p>';
    }
    
    /**
     * Checkbox Feld Callback
     */
    public function checkbox_field_callback($args) {
        $current_language = $this->get_option('language', 'de');
        $field = $args['field'];
        $label = $args['label'];
        $value = $this->get_option($field, false);
        
        // Übersetzungen für Labels
        $translations = [
            'auto_scan' => [
                'de' => 'Bilder automatisch nach C2PA-Inhalten scannen',
                'en' => 'Automatically scan images for C2PA content'
            ],
            'show_comparison' => [
                'de' => 'Vergleich zwischen Basis- und erweiterten Daten anzeigen',
                'en' => 'Show comparison between basic and extended data'
            ],
            'show_raw_data' => [
                'de' => 'Rohdaten-Tab in der Bildanalyse anzeigen',
                'en' => 'Show raw data tab in image analysis'
            ],
            'exclude_featured_images' => [
                'de' => 'Badges für Titelbilder von Seiten und Beiträgen deaktivieren',
                'en' => 'Disable badges for featured images of pages and posts'
            ],
            'disable_notifications' => [
                'de' => 'Benachrichtigungen für Content Credentials und Metadaten deaktivieren',
                'en' => 'Disable notifications for Content Credentials and metadata'
            ]
        ];
        
        $display_label = isset($translations[$field][$current_language]) ? $translations[$field][$current_language] : $label;
        
        echo '<label><input type="checkbox" name="wp_c2pa_combined_settings[' . esc_attr($field) . ']" value="1" ' . checked(1, $value, false) . '> ' . esc_html($display_label) . '</label>';
    }
    
    /**
     * Select Feld Callback
     */
    public function select_field_callback($args) {
        $current_language = $this->get_option('language', 'de');
        $field = $args['field'];
        $options = $args['options'];
        $value = $this->get_option($field, 'bottom-left');
        
        // Übersetzungen für Select-Optionen
        $option_translations = [
            'badge_display' => [
                'icon_only' => [
                    'de' => 'Nur Symbol',
                    'en' => 'Icon Only'
                ],
                'text_only' => [
                    'de' => 'Nur Text',
                    'en' => 'Text Only'
                ],
                'icon_text' => [
                    'de' => 'Symbol und Text',
                    'en' => 'Icon and Text'
                ]
            ],
            'badge_position' => [
                'bottom-left' => [
                    'de' => 'Unten links',
                    'en' => 'Bottom Left'
                ],
                'bottom-right' => [
                    'de' => 'Unten rechts',
                    'en' => 'Bottom Right'
                ],
                'top-left' => [
                    'de' => 'Oben links',
                    'en' => 'Top Left'
                ],
                'top-right' => [
                    'de' => 'Oben rechts',
                    'en' => 'Top Right'
                ]
            ],
            'badge_size' => [
                'small' => [
                    'de' => 'Klein',
                    'en' => 'Small'
                ],
                'medium' => [
                    'de' => 'Mittel',
                    'en' => 'Medium'
                ],
                'large' => [
                    'de' => 'Groß',
                    'en' => 'Large'
                ]
            ],
            'badge_style' => [
                'solid' => [
                    'de' => 'Ausgefüllt',
                    'en' => 'Solid'
                ],
                'outline' => [
                    'de' => 'Rahmen',
                    'en' => 'Outline'
                ],
                'minimal' => [
                    'de' => 'Minimal',
                    'en' => 'Minimal'
                ]
            ],
            'badge_opacity' => [
                '0.7' => [
                    'de' => '70% (transparent)',
                    'en' => '70% (transparent)'
                ],
                '0.8' => [
                    'de' => '80%',
                    'en' => '80%'
                ],
                '0.9' => [
                    'de' => '90%',
                    'en' => '90%'
                ],
                '1.0' => [
                    'de' => '100% (undurchsichtig)',
                    'en' => '100% (opaque)'
                ]
            ],
            'badge_shadow' => [
                'none' => [
                    'de' => 'Kein Schatten',
                    'en' => 'No Shadow'
                ],
                'light' => [
                    'de' => 'Leichter Schatten',
                    'en' => 'Light Shadow'
                ],
                'medium' => [
                    'de' => 'Mittlerer Schatten',
                    'en' => 'Medium Shadow'
                ],
                'strong' => [
                    'de' => 'Starker Schatten',
                    'en' => 'Strong Shadow'
                ]
            ],
            'badge_border_radius' => [
                '0' => [
                    'de' => 'Eckig',
                    'en' => 'Square'
                ],
                '4' => [
                    'de' => 'Leicht gerundet',
                    'en' => 'Slightly Rounded'
                ],
                '8' => [
                    'de' => 'Gerundet',
                    'en' => 'Rounded'
                ],
                '12' => [
                    'de' => 'Stark gerundet',
                    'en' => 'Strongly Rounded'
                ],
                '20' => [
                    'de' => 'Sehr rund',
                    'en' => 'Very Rounded'
                ]
            ],
            'badge_padding' => [
                '4' => [
                    'de' => 'Kompakt (4px)',
                    'en' => 'Compact (4px)'
                ],
                '6' => [
                    'de' => 'Klein (6px)',
                    'en' => 'Small (6px)'
                ],
                '8' => [
                    'de' => 'Standard (8px)',
                    'en' => 'Standard (8px)'
                ],
                '12' => [
                    'de' => 'Groß (12px)',
                    'en' => 'Large (12px)'
                ],
                '16' => [
                    'de' => 'Sehr groß (16px)',
                    'en' => 'Very Large (16px)'
                ]
            ],
            'badge_font_size' => [
                '10' => [
                    'de' => 'Sehr klein (10px)',
                    'en' => 'Very Small (10px)'
                ],
                '12' => [
                    'de' => 'Klein (12px)',
                    'en' => 'Small (12px)'
                ],
                '14' => [
                    'de' => 'Standard (14px)',
                    'en' => 'Standard (14px)'
                ],
                '16' => [
                    'de' => 'Groß (16px)',
                    'en' => 'Large (16px)'
                ],
                '18' => [
                    'de' => 'Sehr groß (18px)',
                    'en' => 'Very Large (18px)'
                ]
            ],
            'badge_font_weight' => [
                'normal' => [
                    'de' => 'Normal',
                    'en' => 'Normal'
                ],
                '500' => [
                    'de' => 'Mittel',
                    'en' => 'Medium'
                ],
                '600' => [
                    'de' => 'Semi-Bold',
                    'en' => 'Semi-Bold'
                ],
                '700' => [
                    'de' => 'Bold',
                    'en' => 'Bold'
                ],
                '800' => [
                    'de' => 'Extra Bold',
                    'en' => 'Extra Bold'
                ]
            ]
        ];
        
        echo '<select name="wp_c2pa_combined_settings[' . esc_attr($field) . ']" class="badge-setting" data-setting="' . esc_attr($field) . '">';
        foreach ($options as $key => $label) {
            // Übersetzte Option verwenden, falls verfügbar
            $display_label = $label;
            if (isset($option_translations[$field][$key][$current_language])) {
                $display_label = $option_translations[$field][$key][$current_language];
            }
            
            echo '<option value="' . esc_attr($key) . '" ' . selected($key, $value, false) . '>' . esc_html($display_label) . '</option>';
        }
        echo '</select>';
    }
    
    /**
     * Text Feld Callback
     */
    public function text_field_callback($args) {
        $field = $args['field'];
        $default = $args['default'] ?? '';
        $description = $args['description'] ?? '';
        $value = $this->get_option($field, $default);
        echo '<input type="text" name="wp_c2pa_combined_settings[' . esc_attr($field) . ']" value="' . esc_attr($value) . '" class="regular-text badge-setting" data-setting="' . esc_attr($field) . '">';
        if ($description) {
            echo '<p class="description">' . esc_html($description) . '</p>';
        }
    }
    
    /**
     * Bedingtes Text-Feld Callback (nur erforderlich bei bestimmten Badge-Display-Optionen)
     */
    public function conditional_text_field_callback($args) {
        $field = $args['field'];
        $value = $this->get_option($field, $args['default'] ?? '');
        $description = $args['description'] ?? '';
        $current_display = $this->get_option('badge_display', 'icon_text');
        
        // Feld nur anzeigen wenn nicht "Nur Symbol" ausgewählt ist
        if ($current_display !== 'icon_only') {
            echo '<input type="text" name="wp_c2pa_combined_settings[' . esc_attr($field) . ']" value="' . esc_attr($value) . '" class="regular-text badge-setting" data-setting="' . esc_attr($field) . '">';
            if ($description) {
                echo '<p class="description">' . esc_html($description) . '</p>';
            }
        } else {
            // Verstecktes Feld für "Nur Symbol" um Validierung zu umgehen
            echo '<input type="hidden" name="wp_c2pa_combined_settings[' . esc_attr($field) . ']" value="Content Credentials">';
            echo '<p class="description" style="color: #666; font-style: italic;">Textfeld ausgeblendet - nicht erforderlich bei "Nur Symbol"</p>';
        }
    }
    
    /**
     * Farb Feld Callback
     */
    public function color_field_callback($args) {
        $field = $args['field'];
        $default = $args['default'] ?? '#0073aa';
        $value = $this->get_option($field, $default);
        echo '<input type="text" name="wp_c2pa_combined_settings[' . esc_attr($field) . ']" value="' . esc_attr($value) . '" class="color-picker badge-setting" data-setting="' . esc_attr($field) . '">';
    }
    
    /**
     * Textarea Feld Callback
     */
    public function textarea_field_callback($args) {
        $current_language = $this->get_option('language', 'de');
        $field = $args['field'];
        $description = $args['description'] ?? '';
        $value = $this->get_option($field, '');
        
        // Übersetzungen für Textarea-Beschreibungen
        $description_translations = [
            'exclude_pages' => [
                'de' => 'Eine URL pro Zeile. Badges werden auf diesen Seiten nicht angezeigt. Beispiel: /kontakt/, /impressum/',
                'en' => 'One URL per line. Badges will not be displayed on these pages. Example: /contact/, /imprint/'
            ]
        ];
        
        $display_description = $description;
        if (isset($description_translations[$field][$current_language])) {
            $display_description = $description_translations[$field][$current_language];
        }
        
        echo '<textarea name="wp_c2pa_combined_settings[' . esc_attr($field) . ']" rows="5" cols="50" class="large-text">' . esc_textarea($value) . '</textarea>';
        if ($display_description) {
            echo '<p class="description">' . esc_html($display_description) . '</p>';
        }
    }
    
    /**
     * Checkbox Liste Feld Callback
     */
    public function checkbox_list_field_callback($args) {
        $current_language = $this->get_option('language', 'de');
        $field = $args['field'];
        $options = $args['options'];
        $saved_values = $this->get_option($field, []);
        
        if (!is_array($saved_values)) {
            $saved_values = [];
        }
        
        // Übersetzungen für Checkbox-Listen-Optionen
        $checkbox_translations = [
            'exclude_post_types' => [
                'page' => [
                    'de' => 'Seiten',
                    'en' => 'Pages'
                ],
                'post' => [
                    'de' => 'Beiträge',
                    'en' => 'Posts'
                ],
                'attachment' => [
                    'de' => 'Medien',
                    'en' => 'Media'
                ],
                'product' => [
                    'de' => 'Produkte (WooCommerce)',
                    'en' => 'Products (WooCommerce)'
                ],
                'shop_order' => [
                    'de' => 'Bestellungen (WooCommerce)',
                    'en' => 'Orders (WooCommerce)'
                ]
            ]
        ];
        
        foreach ($options as $key => $label) {
            $checked = in_array($key, $saved_values) ? 'checked' : '';
            
            // Übersetzte Option verwenden, falls verfügbar
            $display_label = $label;
            if (isset($checkbox_translations[$field][$key][$current_language])) {
                $display_label = $checkbox_translations[$field][$key][$current_language];
            }
            
            echo '<label style="display: block; margin-bottom: 8px;">';
            echo '<input type="checkbox" name="wp_c2pa_combined_settings[' . esc_attr($field) . '][]" value="' . esc_attr($key) . '" ' . $checked . '> ';
            echo esc_html($display_label);
            echo '</label>';
        }
    }
    
    /**
     * Icon Feld Callback
     */
    public function icon_field_callback($args) {
        $field = $args['field'];
        $value = $this->get_option($field, 'shield');
        
        $icons = [
            'shield' => '🛡️',
            'check' => '✅',
            'lock' => '🔒',
            'star' => '⭐',
            'verified' => '✓',
            'certificate' => '📜',
            'eye' => '👁️',
            'info' => 'ℹ️',
            'badge' => '🏷️',
            'security' => '🔐',
            'eye-shield' => '<img src="' . plugin_dir_url(__FILE__) . 'assets/images/eye_shield_badge_transparent.svg" style="width: 20px; height: 20px; vertical-align: middle;" alt="Eye Shield">',
            'custom-png' => '🛡️' // Fallback zu Shield-Icon
        ];
        
        echo '<div class="icon-selector" style="display: flex; flex-wrap: wrap; gap: 6px; margin-top: 8px;">';
        foreach ($icons as $key => $label) {
            $checked = ($value === $key) ? 'checked' : '';
            $selected_style = ($value === $key) ? 'background: #ce8806; color: white; transform: scale(1.1);' : 'background: #f8f9fa; color: #333;';
            echo '<label style="display: inline-flex; align-items: center; justify-content: center; cursor: pointer; transition: all 0.2s ease; border-radius: 8px; border: 2px solid #e9ecef; ' . $selected_style . '">';
            echo '<input type="radio" name="wp_c2pa_combined_settings[' . esc_attr($field) . ']" value="' . esc_attr($key) . '" ' . $checked . ' class="badge-setting" data-setting="' . esc_attr($field) . '" style="position: absolute; opacity: 0; pointer-events: none;"> ';
            
            // Für SVG-Icons HTML verwenden, für Emojis Text
            if ($key === 'eye-shield') {
                echo '<span style="font-size: 18px; padding: 8px; width: 36px; height: 36px; display: flex; align-items: center; justify-content: center; border-radius: 6px;">' . $label . '</span>';
            } else {
                echo '<span style="font-size: 18px; padding: 8px; width: 36px; height: 36px; display: flex; align-items: center; justify-content: center; border-radius: 6px;">' . esc_html($label) . '</span>';
            }
            
            echo '</label>';
        }
        echo '</div>';
        echo '<script>
        jQuery(document).ready(function($) {
            $(".icon-selector label").on("click", function() {
                var $radio = $(this).find("input[type=radio]");
                $radio.prop("checked", true).trigger("change");
                
                // Update visual selection
                $(".icon-selector label").css({
                    "background": "#f8f9fa",
                    "color": "#333",
                    "transform": "scale(1)"
                });
                $(this).css({
                    "background": "#ce8806",
                    "color": "white",
                    "transform": "scale(1.1)"
                });
            });
        });
        </script>';
    }
    
    /**
     * Admin-Seite rendern
     */
    public function admin_page() {
        if (!current_user_can('manage_options')) {
            return;
        }
        
        ?>
        <div class="wrap wp-c2pa-admin">
                            <div class="wp-c2pa-admin-header">
                    <div class="wp-c2pa-admin-title">
                        <div class="wp-c2pa-admin-icon">
                            <img src="<?php echo plugin_dir_url(__FILE__); ?>assets/images/eye_shield_badge_transparent.svg" style="width: 48px; height: 48px; filter: brightness(0) invert(1);" alt="Eye Shield Logo">
                        </div>
                        <div>
                            <h1>C2PA Content Credentials Viewer</h1>
                            <p>Content Credentials Plugin - Konfiguration</p>
                        </div>
                    </div>
                    <div class="wp-c2pa-admin-version">
                        Version <?php echo WP_C2PA_COMBINED_VERSION; ?>
                    </div>
                </div>
            
            <div class="wp-c2pa-admin-body">
                <div class="wp-c2pa-admin-main">
                    <form method="post" action="options.php" id="c2pa-settings-form">
                        <?php settings_fields('wp_c2pa_combined_options'); ?>
                        
                        <div class="wp-c2pa-admin-section">
                            <h2>Haupteinstellungen</h2>
                            <?php do_settings_sections('wp-c2pa-viewer-combined'); ?>
                        </div>
                        
                        <div class="wp-c2pa-admin-actions">
                            <?php submit_button('Einstellungen speichern', 'primary', 'submit', false); ?>
                            <button type="button" id="reset-settings" class="button">Zurücksetzen</button>
                        </div>
                        
                        <script>
                        jQuery(document).ready(function($) {
                            // Collapsible functionality for C2PA Resources
                            $('.collapsible-header').on('click', function() {
                                var $content = $(this).siblings('.collapsible-content');
                                var $icon = $(this).find('.toggle-icon');
                                
                                if ($content.is(':visible')) {
                                    $content.slideUp(300);
                                    $icon.text('▼').css('transform', 'rotate(0deg)');
                                } else {
                                    $content.slideDown(300);
                                    $icon.text('▲').css('transform', 'rotate(180deg)');
                                }
                            });
                            
                            // Button translations based on language setting
                            var currentLanguage = '<?php echo $this->get_option('language', 'de'); ?>';
                            
                            if (currentLanguage === 'en') {
                                // Translate Save Settings button
                                $('input[type="submit"][value="Einstellungen speichern"]').val('Save Settings');
                                
                                // Translate Reset button
                                $('#reset-settings').text('Reset');
                                
                                // Update page title and description
                                $('h1').text('C2PA Content Credentials Viewer');
                                $('.wp-c2pa-admin-title p').text('Content Credentials Plugin - Configuration');
                                
                                                // Update section titles - entferne doppelte Main Settings
                // $('h2').text('Main Settings'); // Entfernt um doppelte Anzeige zu vermeiden
                                
                                // Update status items
                                $('.wp-c2pa-status-item span:last').each(function() {
                                    var text = $(this).text();
                                    if (text === 'Plugin aktiv') {
                                        $(this).text('Plugin active');
                                    } else if (text === 'C2PA Bibliothek verfügbar') {
                                        $(this).text('C2PA library available');
                                    } else if (text === 'Auto-Scan aktiviert') {
                                        $(this).text('Auto-Scan enabled');
                                    }
                                });
                                
                                // Update sidebar content
                                $('.wp-c2pa-admin-card h3').each(function() {
                                    var text = $(this).text();
                                    if (text === 'Plugin Status') {
                                        $(this).text('Plugin Status');
                                    } else if (text === 'C2PA Ressourcen & Links') {
                                        $(this).text('C2PA Resources & Links');
                                    }
                                });
                                
                                // Update collapsible content
                                $('.collapsible-content p').each(function() {
                                    var text = $(this).text();
                                    if (text.includes('Wichtige Links und Ressourcen rund um C2PA')) {
                                        $(this).text('Important links and resources around C2PA and Content Credentials:');
                                    }
                                });
                                
                                $('.collapsible-content h4').each(function() {
                                    var text = $(this).text();
                                    if (text === 'Offizielle C2PA Ressourcen') {
                                        $(this).text('Official C2PA Resources');
                                    } else if (text === 'Content Authenticity Initiative (CAI)') {
                                        $(this).text('Content Authenticity Initiative (CAI)');
                                    }
                                });
                                
                                // Translate field labels and descriptions
                                $('label').each(function() {
                                    var text = $(this).text();
                                    if (text === 'Die Version der C2PA-Bibliothek, die verwendet werden soll.') {
                                        $(this).text('The version of the C2PA library to be used.');
                                    }
                                });
                                
                                // Translate field descriptions
                                $('.description').each(function() {
                                    var text = $(this).text();
                                    if (text === 'Textfeld ausgeblendet - nicht erforderlich bei "Nur Symbol"') {
                                        $(this).text('Text field hidden - not required for "Icon Only"');
                                    }
                                });
                                
                                // Translate section descriptions
                                $('p').each(function() {
                                    var text = $(this).text();
                                    if (text === 'Passen Sie das Aussehen und Verhalten der Content Credentials Badges an.') {
                                        $(this).text('Customize the appearance and behavior of Content Credentials badges.');
                                    } else if (text === 'Konfigurieren Sie auf welchen Seiten die Content Credentials Badges angezeigt werden sollen.') {
                                        $(this).text('Configure on which pages the Content Credentials badges should be displayed.');
                                    } else if (text === 'Shortcuts: Strg+S (Speichern), Strg+R (Zurücksetzen)') {
                                        $(this).text('Shortcuts: Ctrl+S (Save), Ctrl+R (Reset)');
                                    }
                                });
                                
                                // Translate plugin info section
                                $('.wp-c2pa-admin-card').each(function() {
                                    var $card = $(this);
                                    
                                    // Find and translate "Über das Plugin" section
                                    $card.find('h3').each(function() {
                                        var text = $(this).text();
                                        if (text === 'Über das Plugin') {
                                            $(this).text('About the Plugin');
                                        }
                                    });
                                    
                                    // Translate plugin description
                                    $card.find('p').each(function() {
                                        var text = $(this).text();
                                        if (text.includes('Das C2PA Content Credentials Viewer Plugin kombiniert zwei Ansätze:')) {
                                            $(this).text('The C2PA Content Credentials Viewer Plugin combines two approaches:');
                                        }
                                    });
                                    
                                    // Translate feature list
                                    $card.find('.feature-item strong').each(function() {
                                        var text = $(this).text();
                                        if (text === 'Tab 1: Basis C2PA-Daten') {
                                            $(this).text('Tab 1: Basic C2PA Data');
                                        } else if (text === 'Tab 2: Erweiterte Analyse') {
                                            $(this).text('Tab 2: Extended Analysis');
                                        }
                                    });
                                    
                                    // Translate feature list items
                                    $card.find('.feature-item ul li').each(function() {
                                        var text = $(this).text();
                                        if (text === 'Nur C2PA-Manifeste') {
                                            $(this).text('C2PA manifests only');
                                        } else if (text === 'Standardkonforme Anzeige') {
                                            $(this).text('Standard-compliant display');
                                        } else if (text === 'Änderungen & Creator-Info') {
                                            $(this).text('Changes & Creator Info');
                                        } else if (text === 'EXIF/IPTC/XMP Metadaten') {
                                            $(this).text('EXIF/IPTC/XMP metadata');
                                        } else if (text === 'KI-Herkunftserkennung') {
                                            $(this).text('AI origin detection');
                                        } else if (text === 'Technische Rohdaten') {
                                            $(this).text('Technical raw data');
                                        } else if (text === 'Video-Unterstützung') {
                                            $(this).text('Video support');
                                        }
                                    });
                                    
                                    // Translate shortcode section
                                    $card.find('h3').each(function() {
                                        var text = $(this).text();
                                        if (text === '🎯 Shortcode') {
                                            $(this).text('🎯 Shortcode');
                                        }
                                    });
                                    
                                    $card.find('p').each(function() {
                                        var text = $(this).text();
                                        if (text.includes('Verwenden Sie diesen Shortcode für manuelle Einbindung:')) {
                                            $(this).text('Use this shortcode for manual integration:');
                                        } else if (text.includes('Oder mit URL:')) {
                                            $(this).text('Or with URL:');
                                        }
                                    });
                                });
                            }
                        });
                        </script>
                    </form>
                </div>
                
                <div class="wp-c2pa-admin-sidebar">
                    <div class="wp-c2pa-admin-card">
                        <h3>Plugin Status</h3>
                        <div class="wp-c2pa-status-item">
                            <span class="status-indicator active"></span>
                            <span><?php echo $this->get_option('language', 'de') === 'en' ? 'Plugin active' : 'Plugin aktiv'; ?></span>
                        </div>
                        <div class="wp-c2pa-status-item">
                            <span class="status-indicator active"></span>
                            <span><?php echo $this->get_option('language', 'de') === 'en' ? 'C2PA library available' : 'C2PA Bibliothek verfügbar'; ?></span>
                        </div>
                        <div class="wp-c2pa-status-item">
                            <span class="status-indicator active"></span>
                            <span>Auto-Scan aktiviert</span>
                        </div>
                    </div>
                    
                    <div class="wp-c2pa-admin-card collapsible">
                        <h3 class="collapsible-header" style="cursor: pointer; display: flex; align-items: center; justify-content: space-between;">
                            C2PA Ressourcen & Links
                            <span class="toggle-icon" style="font-size: 12px; transition: transform 0.3s ease;">▼</span>
                        </h3>
                        <div class="collapsible-content" style="display: none; margin-top: 15px;">
                            <p style="margin-bottom: 15px; font-size: 13px; color: #666;">Wichtige Links und Ressourcen rund um C2PA und Content Credentials:</p>
                            
                            <!-- Offizielle C2PA Ressourcen -->
                            <div style="margin-bottom: 20px;">
                                <h4 style="font-size: 14px; margin-bottom: 8px; color: #333;">Offizielle C2PA Ressourcen</h4>
                                <ul style="list-style: none; padding-left: 0; margin: 0;">
                                    <li style="margin-bottom: 6px;"><a href="https://c2pa.org" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA.org - Offizielle Website</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://c2pa.org/specifications/" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA Spezifikationen</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://c2pa.org/public-draft/" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA Public Draft</a></li>
                                </ul>
                            </div>
                            
                            <!-- Content Authenticity Initiative (CAI) -->
                            <div style="margin-bottom: 20px;">
                                <h4 style="font-size: 14px; margin-bottom: 8px; color: #333;">Content Authenticity Initiative (CAI)</h4>
                                <ul style="list-style: none; padding-left: 0; margin: 0;">
                                    <li style="margin-bottom: 6px;"><a href="https://contentauthenticity.org" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">CAI - Adobe Initiative</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://contentauthenticity.org/developers" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">CAI Developer Resources</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://contentauthenticity.org/verify" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">CAI Verify Tool</a></li>
                                </ul>
                            </div>
                            
                            <!-- Technische Bibliotheken -->
                            <div style="margin-bottom: 20px;">
                                <h4 style="font-size: 14px; margin-bottom: 8px; color: #333;">Technische Bibliotheken</h4>
                                <ul style="list-style: none; padding-left: 0; margin: 0;">
                                    <li style="margin-bottom: 6px;"><a href="https://github.com/contentauth/c2pa-js" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA JavaScript Library (GitHub)</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://www.npmjs.com/package/c2pa" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA NPM Package</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://github.com/contentauth/c2pa-rs" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA Rust Library</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://github.com/contentauth/c2pa-python" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA Python Library</a></li>
                                </ul>
                            </div>
                            
                            <!-- Tools und Validierung -->
                            <div style="margin-bottom: 20px;">
                                <h4 style="font-size: 14px; margin-bottom: 8px; color: #333;">Tools & Validierung</h4>
                                <ul style="list-style: none; padding-left: 0; margin: 0;">
                                    <li style="margin-bottom: 6px;"><a href="https://verify.contentauthenticity.org" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">CAI Verify Online Tool</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://github.com/contentauth/c2pa-js/tree/main/tools" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA Command Line Tools</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://c2pa.org/playground/" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA Playground</a></li>
                                </ul>
                            </div>
                            
                            <!-- Dokumentation und Guides -->
                            <div style="margin-bottom: 20px;">
                                <h4 style="font-size: 14px; margin-bottom: 8px; color: #333;">Dokumentation & Guides</h4>
                                <ul style="list-style: none; padding-left: 0; margin: 0;">
                                    <li style="margin-bottom: 6px;"><a href="https://c2pa.org/developers/" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA Developer Guide</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://c2pa.org/use-cases/" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA Use Cases</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://c2pa.com/faq/" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA FAQ</a></li>
                                </ul>
                            </div>
                            
                            <!-- Community und Support -->
                            <div style="margin-bottom: 20px;">
                                <h4 style="font-size: 14px; margin-bottom: 8px; color: #333;">Community & Support</h4>
                                <ul style="list-style: none; padding-left: 0; margin: 0;">
                                    <li style="margin-bottom: 6px;"><a href="https://github.com/contentauth" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">CAI GitHub Organization</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://discord.gg/contentauthenticity" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">CAI Discord Community</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://c2pa.org/contact/" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA Kontakt</a></li>
                                </ul>
                            </div>
                            
                            <!-- Aktuelle Versionen -->
                            <div style="margin-bottom: 20px;">
                                <h4 style="font-size: 14px; margin-bottom: 8px; color: #333;">Aktuelle Versionen</h4>
                                <ul style="list-style: none; padding-left: 0; margin: 0;">
                                    <li style="margin-bottom: 6px;"><a href="https://github.com/contentauth/c2pa-js/releases" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">C2PA-JS Releases</a></li>
                                    <li style="margin-bottom: 6px;"><a href="https://www.npmjs.com/package/c2pa" target="_blank" style="text-decoration: none; color: #0073aa; font-size: 12px;">NPM Latest Version</a></li>
                                </ul>
                            </div>
                        </div>
                    </div>
                    
                    <div class="wp-c2pa-admin-card">
                        <h3>Über das Plugin</h3>
                        <p>Das C2PA Content Credentials Viewer Plugin kombiniert zwei Ansätze:</p>
                        
                        <div class="feature-list">
                            <div class="feature-item">
                                <strong>Tab 1: Basis C2PA-Daten</strong>
                                <ul>
                                    <li>Nur C2PA-Manifeste</li>
                                    <li>Standardkonforme Anzeige</li>
                                    <li>Änderungen & Creator-Info</li>
                                </ul>
                            </div>
                            
                            <div class="feature-item">
                                <strong>Tab 2: Erweiterte Analyse</strong>
                                <ul>
                                    <li>EXIF/IPTC/XMP Metadaten</li>
                                    <li>KI-Herkunftserkennung</li>
                                    <li>Technische Rohdaten</li>
                                    <li>Video-Unterstützung</li>
                                </ul>
                            </div>
                        </div>
                    </div>
                    
                    <div class="wp-c2pa-admin-card">
                        <h3>🎯 Shortcode</h3>
                        <p><?php echo $this->get_option('language', 'de') === 'en' ? 'Use this shortcode for manual integration:' : 'Verwenden Sie diesen Shortcode für manuelle Einbindung:'; ?></p>
                        <code>[c2pa_viewer_combined image_id="123"]</code>
                        <p style="margin-top: 10px;">
                            <small><?php echo $this->get_option('language', 'de') === 'en' ? 'Or with URL:' : 'Oder mit URL:'; ?> <code>[c2pa_viewer_combined image_url="..."]</code></small>
                        </p>
                    </div>
                </div>
            </div>
        </div>
        <?php
    }
    
    /**
     * Einstellungen abrufen
     */
    private function get_option($key, $default = null) {
        $options = get_option('wp_c2pa_combined_settings', []);
        return isset($options[$key]) ? $options[$key] : $default;
    }
    
    /**
     * Assets laden prüfen
     */
    private function should_load_assets() {
        // Im Admin-Bereich nicht laden
        if (is_admin()) {
            return false;
        }
        
        // Prüfen ob Badges auf dieser Seite angezeigt werden sollen
        if (!$this->should_show_badges_on_current_page()) {
            return false;
        }
        
        // Nur auf Seiten mit Inhalt laden
        if (is_singular() || is_archive() || is_home()) {
            return true;
        }
        
        return false;
    }
    
    /**
     * Prüfen ob Badges auf der aktuellen Seite angezeigt werden sollen
     */
    private function should_show_badges_on_current_page() {
        // Ausgeschlossene Seiten prüfen
        $exclude_pages = $this->get_option('exclude_pages', '');
        if (!empty($exclude_pages)) {
            $exclude_urls = array_filter(array_map('trim', explode("\n", $exclude_pages)));
            $current_url = $_SERVER['REQUEST_URI'] ?? '';
            
            foreach ($exclude_urls as $exclude_url) {
                if (strpos($current_url, $exclude_url) !== false) {
                    return false;
                }
            }
        }
        
        // Ausgeschlossene Beitragstypen prüfen
        $exclude_post_types = $this->get_option('exclude_post_types', []);
        if (!empty($exclude_post_types) && is_array($exclude_post_types)) {
            if (is_singular() && in_array(get_post_type(), $exclude_post_types)) {
                return false;
            }
        }
        
        // Titelbilder ausschließen prüfen
        $exclude_featured_images = $this->get_option('exclude_featured_images', false);
        if ($exclude_featured_images && is_singular()) {
            // Prüfen ob das aktuelle Bild ein Titelbild ist
            $current_post_id = get_the_ID();
            $featured_image_id = get_post_thumbnail_id($current_post_id);
            
            // Wenn wir ein Bild verarbeiten und es das Titelbild ist, Badge ausschließen
            if ($featured_image_id && isset($_POST['image_id']) && $_POST['image_id'] == $featured_image_id) {
                return false;
            }
        }
        
        return true;
    }
    
    /**
     * Shortcode Handler
     */
    public function shortcode_handler($atts) {
        $atts = shortcode_atts([
            'image_id' => '',
            'image_url' => '',
            'position' => 'bottom-left'
        ], $atts);
        
        if (empty($atts['image_id']) && empty($atts['image_url'])) {
            return '<p>Bitte geben Sie eine Bild-ID oder URL an.</p>';
        }
        
        $output = '<div class="wp-c2pa-combined-manual-viewer" data-position="' . esc_attr($atts['position']) . '">';
        
        if (!empty($atts['image_id'])) {
            $image = wp_get_attachment_image($atts['image_id'], 'full', false, ['class' => 'wp-c2pa-combined-image']);
            $output .= $image;
        } else {
            $output .= '<img src="' . esc_url($atts['image_url']) . '" class="wp-c2pa-combined-image" alt="">';
        }
        
        $output .= '</div>';
        
        return $output;
    }
    

    
    /**
     * Plugin aktivieren
     */
    public function activate() {
        // Standardeinstellungen setzen
        $default_settings = [
            'language' => 'de',
            'c2pa_version' => '0.30.11',
            'auto_scan' => true,
            'badge_position' => 'bottom-left',
            'show_loading' => true,
            'show_comparison' => true,
            'show_raw_data' => false,
            'badge_display' => 'icon_text',
            'badge_text' => 'Content Credentials',
            'badge_color' => '#0073aa',
            'badge_icon' => 'shield',
            'badge_size' => 'medium',
            'badge_style' => 'solid',
            // 'badge_opacity' => '1.0', // DEAKTIVIERT
            'badge_shadow' => 'light',
            'badge_border_radius' => '8',
            'badge_padding' => '8',
            'badge_font_size' => '14',
            'badge_font_weight' => '600',
            'exclude_pages' => '',
            'exclude_post_types' => [],
            'exclude_featured_images' => false
        ];
        
        add_option('wp_c2pa_combined_settings', $default_settings);
        
        // Flush Rewrite Rules
        flush_rewrite_rules();
    }
    

    
    /**
     * Bild-Skalierung deaktivieren
     */
    public function disable_image_scaling($threshold, $imagesize, $file, $attachment_id) {
        // Deaktiviere das automatische Skalieren von Bildern
        return false;
    }
    
    /**
     * Custom Badge Handler (PNG oder SVG)
     */
    private function get_custom_png_badge() {
        // Prüfe zuerst PNG
        $custom_badge_png_path = WP_C2PA_COMBINED_PATH . 'assets/images/custom-badge.png';
        $custom_badge_png_url = WP_C2PA_COMBINED_URL . 'assets/images/custom-badge.png';
        
        if (file_exists($custom_badge_png_path)) {
            return '<img src="' . esc_url($custom_badge_png_url) . '" style="width: 16px; height: 16px; vertical-align: middle;" alt="Custom Badge">';
        }
        
        // Prüfe SVG
        $custom_badge_svg_path = WP_C2PA_COMBINED_PATH . 'assets/images/custom-badge.svg';
        $custom_badge_svg_url = WP_C2PA_COMBINED_URL . 'assets/images/custom-badge.svg';
        
        if (file_exists($custom_badge_svg_path)) {
            return '<img src="' . esc_url($custom_badge_svg_url) . '" style="width: 16px; height: 16px; vertical-align: middle;" alt="Custom Badge">';
        }
        
        // Fallback zu Shield-Icon wenn keine Datei existiert
        return '🛡️';
    }
    
    /**
     * Plugin deaktivieren
     */
    public function deactivate() {
        // Flush Rewrite Rules
        flush_rewrite_rules();
    }
}

// Plugin initialisieren
new WP_C2PA_Viewer_Combined();
